<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\User;
use Carbon\Carbon;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Txn;

class AddDailyProfit extends Command
{
    protected $signature = 'invest:add-daily-profit';
    protected $description = 'Add daily profit to users based on investment plans';

    public function handle()
    {
        $timezone = setting('site_timezone', 'global') ?? 'UTC';
        $now = Carbon::now($timezone);
        
        $this->info("Starting profit distribution at: " . $now->toDateTimeString());

        // Get all ongoing investments where profit time has passed
        $investments = Invest::with(['user', 'schema'])
            ->where('status', 'ongoing') // Make sure status matches your enum
            ->where('next_profit_time', '<=', $now)
            ->get();

        $this->info("Found " . $investments->count() . " investments ready for profit distribution");

        $processedCount = 0;
        $errorCount = 0;

        foreach ($investments as $invest) {
            try {
                $user = $invest->user;

                if (!$user || !$user->exists) {
                    $this->error("Investment ID {$invest->id} has no valid user, skipping...");
                    $errorCount++;
                    continue;
                }

                // Calculate profit based on dynamic ROI from schema
                $profitAmount = $invest->calculated_profit;

                // Add profit to user's profit balance
                $user->increment('profit_balance', $profitAmount);

                // Update investment record
                $invest->increment('already_return_profit');
                $invest->last_profit_time = $now;
                
                // Calculate next profit time based on hours schedule
                $periodHours = optional($invest->schema->schedule)->time
                    ?? ($invest->period_hours ?: 24);
                $nextProfitTime = $now->copy()->addHours($periodHours);
                $invest->next_profit_time = $nextProfitTime;

                // Check if investment should be completed
                if ($invest->return_type == 'period' && $invest->already_return_profit >= $invest->number_of_period) {
                    $invest->status = 'completed';
                    
                    // Return capital if applicable
                    if ($invest->capital_back) {
                        $user->increment('balance', $invest->invest_amount);
                        
                        // Create transaction record for capital return
                        Txn::new(
                            $invest->invest_amount,
                            0,
                            $invest->invest_amount,
                            'system',
                            'Capital Return for ' . $invest->schema->name,
                            TxnType::CapitalReturn,
                            TxnStatus::Success,
                            null,
                            null,
                            $user->id
                        );
                        
                        $this->info("Capital returned for Investment ID {$invest->id}");
                    }
                }

                $invest->save();

                // Create transaction record for profit
                Txn::new(
                    $profitAmount,
                    0,
                    $profitAmount,
                    'system',
                    'Daily Profit from ' . $invest->schema->name,
                    TxnType::Interest,
                    TxnStatus::Success,
                    null,
                    null,
                    $user->id
                );

                $this->info("Profit {$profitAmount} added for Investment ID {$invest->id}, User ID {$user->id}");
                $processedCount++;

            } catch (\Exception $e) {
                $this->error("Error processing Investment ID {$invest->id}: " . $e->getMessage());
                $errorCount++;
                continue;
            }
        }

        $this->info("Profit distribution completed. Processed: {$processedCount}, Errors: {$errorCount}");
        return 0;
    }
}