<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RoiSchedule;
use App\Models\Invest;
use App\Models\User;
use Carbon\Carbon;

class CheckRoiStatus extends Command
{
    protected $signature = 'roi:check-status';
    protected $description = 'Check current ROI status and statistics';

    public function handle()
    {
        $this->info('🔍 Checking ROI Status...');
        $this->newLine();

        // Get all ROI schedules with their status
        $totalSchedules = RoiSchedule::count();
        $pendingSchedules = RoiSchedule::where('status', 'pending')->count();
        $readySchedules = RoiSchedule::where('status', 'ready')->count();
        $sentSchedules = RoiSchedule::where('status', 'sent')->count();
        $completedSchedules = RoiSchedule::where('status', 'completed')->count();

        // Display statistics
        $this->info('📊 ROI Schedule Statistics:');
        $this->table(
            ['Status', 'Count', 'Percentage'],
            [
                ['Total Schedules', $totalSchedules, '100%'],
                ['Pending', $pendingSchedules, $totalSchedules > 0 ? round(($pendingSchedules / $totalSchedules) * 100, 2) . '%' : '0%'],
                ['Ready', $readySchedules, $totalSchedules > 0 ? round(($readySchedules / $totalSchedules) * 100, 2) . '%' : '0%'],
                ['Sent', $sentSchedules, $totalSchedules > 0 ? round(($sentSchedules / $totalSchedules) * 100, 2) . '%' : '0%'],
                ['Completed', $completedSchedules, $totalSchedules > 0 ? round(($completedSchedules / $totalSchedules) * 100, 2) . '%' : '0%'],
            ]
        );

        $this->newLine();

        // Check pending ROI that should be ready
        $expiredPending = RoiSchedule::where('status', 'pending')
            ->where('roi_end_time', '<=', now())
            ->count();

        if ($expiredPending > 0) {
            $this->warn("⚠️  Found {$expiredPending} pending ROI schedules that should be ready!");
        }

        // Check ready ROI that can be collected
        $readyForCollection = RoiSchedule::where('status', 'ready')
            ->where('roi_ready_time', '<=', now())
            ->count();

        if ($readyForCollection > 0) {
            $this->info("✅ Found {$readyForCollection} ROI schedules ready for collection!");
        }

        // Get total pending ROI amount
        $pendingAmount = RoiSchedule::where('status', 'pending')->sum('roi_amount');
        $readyAmount = RoiSchedule::where('status', 'ready')->sum('roi_amount');
        $sentAmount = RoiSchedule::where('status', 'sent')->sum('roi_amount');

        $this->newLine();
        $this->info('💰 ROI Amount Summary:');
        $this->table(
            ['Status', 'Total Amount'],
            [
                ['Pending', '$' . number_format($pendingAmount, 2)],
                ['Ready for Collection', '$' . number_format($readyAmount, 2)],
                ['Already Sent', '$' . number_format($sentAmount, 2)],
            ]
        );

        // Check ongoing investments without ROI schedules
        $ongoingInvestments = Invest::where('status', 'ongoing')->count();
        $investmentsWithSchedules = Invest::where('status', 'ongoing')
            ->whereHas('roiSchedules')
            ->count();
        $investmentsWithoutSchedules = $ongoingInvestments - $investmentsWithSchedules;

        if ($investmentsWithoutSchedules > 0) {
            $this->warn("⚠️  Found {$investmentsWithoutSchedules} ongoing investments without ROI schedules!");
        }

        $this->newLine();
        $this->info('📈 Investment Status:');
        $this->table(
            ['Type', 'Count'],
            [
                ['Total Ongoing Investments', $ongoingInvestments],
                ['With ROI Schedules', $investmentsWithSchedules],
                ['Without ROI Schedules', $investmentsWithoutSchedules],
            ]
        );

        // Show recent ROI schedules
        $this->newLine();
        $this->info('🕒 Recent ROI Schedules (Last 10):');
        $recentSchedules = RoiSchedule::with(['user', 'invest.schema'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        if ($recentSchedules->count() > 0) {
            $tableData = [];
            foreach ($recentSchedules as $schedule) {
                $tableData[] = [
                    $schedule->id,
                    $schedule->user->username ?? 'N/A',
                    $schedule->invest->schema->name ?? 'N/A',
                    '$' . number_format($schedule->roi_amount, 2),
                    $schedule->status,
                    $schedule->roi_cycle . '/' . $schedule->total_cycles,
                    $schedule->created_at->format('M d, Y H:i')
                ];
            }

            $this->table(
                ['ID', 'User', 'Plan', 'Amount', 'Status', 'Cycle', 'Created'],
                $tableData
            );
        } else {
            $this->info('No ROI schedules found.');
        }

        $this->newLine();
        $this->info('✅ ROI Status check completed!');
        
        return Command::SUCCESS;
    }
}
