<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Carbon\Carbon;

class CheckTimezone extends Command
{
    protected $signature = 'timezone:check';
    protected $description = 'Check server and application timezone settings';

    public function handle()
    {
        $this->info('=== Timezone Configuration Check ===');
        
        // Server timezone
        $this->info('Server Timezone: ' . date_default_timezone_get());
        $this->info('Server Time: ' . date('Y-m-d H:i:s'));
        
        // Application timezone
        $this->info('App Timezone: ' . config('app.timezone'));
        $this->info('App Time: ' . Carbon::now()->format('Y-m-d H:i:s'));
        
        // Site timezone setting
        $siteTimezone = setting('site_timezone', 'global');
        $this->info('Site Timezone Setting: ' . $siteTimezone);
        $this->info('Site Time: ' . Carbon::now($siteTimezone)->format('Y-m-d H:i:s'));
        
        // UTC time for reference
        $this->info('UTC Time: ' . Carbon::now('UTC')->format('Y-m-d H:i:s'));
        
        // Check if timezones match
        if (date_default_timezone_get() !== $siteTimezone) {
            $this->warn('⚠️  Server timezone and site timezone do not match!');
            $this->warn('This can cause timer issues.');
        } else {
            $this->info('✅ Server and site timezones match.');
        }
        
        // Check if app timezone matches
        if (config('app.timezone') !== $siteTimezone) {
            $this->warn('⚠️  App timezone and site timezone do not match!');
        } else {
            $this->info('✅ App and site timezones match.');
        }
        
        $this->info('=== End Timezone Check ===');
    }
}
