<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\RoiSchedule;
use Carbon\Carbon;

class CreateExistingRoiSchedules extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'roi:create-existing-schedules';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create ROI schedules for existing investments';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Creating ROI schedules for existing investments...');
        
        // Get all ongoing investments that don't have ROI schedules
        $investments = Invest::with(['user', 'schema'])
            ->where('status', 'ongoing')
            ->whereDoesntHave('roiSchedules')
            ->get();
            
        $createdCount = 0;
        
        foreach ($investments as $investment) {
            // Skip if schema is null
            if (!$investment->schema) {
                $this->warn("Skipping Investment ID: {$investment->id} - Schema not found");
                continue;
            }
            
            // Calculate ROI amount
            $roiAmount = ($investment->invest_amount * $investment->schema->return_interest) / 100;
            
            // Calculate total cycles based on plan's number_of_period
            $totalCycles = $investment->schema->number_of_period ?? 30;
            
            // Create first ROI schedule
            RoiSchedule::create([
                'invest_id' => $investment->id,
                'user_id' => $investment->user_id,
                'schema_id' => $investment->schema_id,
                'invest_amount' => $investment->invest_amount,
                'roi_percentage' => $investment->schema->return_interest,
                'roi_amount' => $roiAmount,
                'roi_start_time' => Carbon::parse($investment->created_at),
                'roi_end_time' => Carbon::parse($investment->created_at)->addHours(24),
                'status' => 'pending',
                'roi_cycle' => 1,
                'total_cycles' => $totalCycles
            ]);
            
            $createdCount++;
            
            $this->line("Created ROI schedule for Investment ID: {$investment->id}, User: {$investment->user->username}");
        }
        
        $this->info("Created {$createdCount} ROI schedules for existing investments.");
        
        return Command::SUCCESS;
    }
}
