<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Support\Str;

class DistributeRankingSalary extends Command
{
    protected $signature = 'salary:distribute';
    protected $description = 'Distribute monthly ranking salary to users';

    public function handle()
    {
        $currentMonth = now()->format('Y-m');
        $totalUsers = 0;
        $eligibleUsers = 0;
        $totalSalaryDistributed = 0;

        // Get only users who have a ranking (ranking_id is not null)
        $users = User::with('rank')
                    ->whereNotNull('ranking_id')
                    ->where('status', true)
                    ->get();

        $this->info("Found {$users->count()} users with rankings.");

        foreach ($users as $user) {
            $totalUsers++;
            
            // Check if user has a valid ranking with salary > 0
            if (!$user->rank || !$user->rank->salary || $user->rank->salary <= 0) {
                continue;
            }

            $salary = $user->rank->salary;

            // Check if salary already distributed this month (prevent duplicates)
            $alreadyDistributed = Transaction::where('user_id', $user->id)
                ->where('type', 'salary')
                ->where('description', 'LIKE', "Monthly ranking salary credited for {$currentMonth}%")
                ->exists();

            if ($alreadyDistributed) {
                $this->warn("Salary already distributed to {$user->username} for {$currentMonth}");
                continue;
            }

            // Distribute salary
            try {
                // 1. Update user balance
                $user->increment('profit_balance', $salary);
                
                // 2. Create transaction record
                Transaction::create([
                    'user_id'       => $user->id,
                    'target_type'   => 'System',
                    'tnx'           => 'SAL' . strtoupper(Str::random(9)), // Salary prefix
                    'description'   => "Monthly ranking salary credited for {$currentMonth} - Rank: {$user->rank->ranking_name}",
                    'amount'        => $salary,
                    'type'          => 'salary',
                    'charge'        => 0,
                    'final_amount'  => $salary,
                    'method'        => 'system',
                    'pay_currency'  => setting('site_currency', 'global'),
                    'pay_amount'    => $salary,
                    'status'        => 'completed',
                    'created_at'    => now(),
                    'updated_at'    => now(),
                ]);

                $eligibleUsers++;
                $totalSalaryDistributed += $salary;
                
                $this->info("✅ Salary distributed to {$user->username}: {$salary} (Rank: {$user->rank->ranking_name})");
                
            } catch (\Exception $e) {
                $this->error("❌ Failed to distribute salary to {$user->username}: " . $e->getMessage());
            }
        }

        $this->info("=====================================");
        $this->info("📊 SALARY DISTRIBUTION SUMMARY");
        $this->info("=====================================");
        $this->info("📅 Month: {$currentMonth}");
        $this->info("👥 Total users checked: {$totalUsers}");
        $this->info("✅ Eligible users: {$eligibleUsers}");
        $this->info("💰 Total salary distributed: {$totalSalaryDistributed}");
        $this->info("=====================================");
    }
}
