<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RecoverMissedProfits extends Command
{
    protected $signature = 'invest:recover-missed-profits';
    protected $description = 'Recover missed profit payments';

    public function handle()
    {
        $now = now();

        $investments = Invest::where('status', 'ongoing')
            ->where('next_profit_time', '<=', $now)
            ->where('interest', '>', 0) // Only process investments with interest
            ->with('user')
            ->get();

        foreach ($investments as $invest) {
            DB::transaction(function () use ($invest, $now) {
                $user = $invest->user;

                if (!$user) {
                    \Log::warning("No user found for investment ID: {$invest->id}");
                    return;
                }

                // Calculate profit
                $profit = ($invest->invest_amount * $invest->interest) / 100;

                // Update user balance
                $user->profit_balance += $profit;
                $user->save();

                // Update investment record
                $invest->already_return_profit += 1;
                $invest->last_profit_time = $now;
                $invest->next_profit_time = $now->copy()->addHours($invest->period_hours);

                // Complete investment if it's the last period
                if (
                    $invest->return_type === 'period' &&
                    $invest->already_return_profit >= $invest->number_of_period
                ) {
                    $invest->status = 'completed';

                    if ($invest->capital_back) {
                        $user->balance += $invest->invest_amount;
                        $user->save();
                    }
                }

                $invest->save();
            });
        }

        $this->info('✅ Missed profits successfully recovered!');
    }
}
