<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ResetAllInvestmentTimers extends Command
{
    protected $signature = 'investment:reset-all-timers {--hours=24 : Hours to set for all timers} {--confirm : Confirm the reset}';
    protected $description = 'Reset all investment timers to a specific time from now';

    public function handle()
    {
        $hours = $this->option('hours');
        $confirmed = $this->option('confirm');
        
        $this->info("=== Reset All Investment Timers ===");
        $this->info("Hours to set: {$hours}");
        
        if (!$confirmed) {
            $this->warn("This will reset ALL ongoing investment timers!");
            $this->warn("Add --confirm flag to proceed.");
            return;
        }
        
        // Get all ongoing investments
        $investments = Invest::where('status', 'ongoing')->get();
        $this->info("Found {$investments->count()} ongoing investments");
        
        if ($investments->count() === 0) {
            $this->info("No ongoing investments found.");
            return;
        }
        
        // Get site timezone
        $siteTimezone = setting('site_timezone', 'global');
        $now = Carbon::now($siteTimezone);
        $newTime = $now->copy()->addHours($hours);
        
        $this->info("Setting all timers to: {$newTime}");
        
        // Update all investments
        $updated = DB::table('invests')
            ->where('status', 'ongoing')
            ->update([
                'next_profit_time' => $newTime,
                'last_profit_time' => $now,
                'period_hours' => $hours,
                'updated_at' => now(),
            ]);
            
        $this->info("✅ Updated {$updated} investments");
        $this->info("All timers reset to {$hours} hours from now");
    }
}
