<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\RoiSchedule;
use App\Models\Schema;

class SetTestRoiPeriod extends Command
{
    protected $signature = 'roi:set-test-period {username} {minutes=1}';
    protected $description = 'Set ROI period to 1 minute for testing specific user';

    public function handle()
    {
        $username = $this->argument('username');
        $minutes = $this->argument('minutes');
        
        $this->info("🔧 Setting ROI period to {$minutes} minute(s) for user: {$username}");
        
        // Find user
        $user = User::where('username', $username)->first();
        
        if (!$user) {
            $this->error("❌ User '{$username}' not found!");
            return Command::FAILURE;
        }
        
        $this->info("✅ User found: {$user->first_name} {$user->last_name}");
        
        // Get user's active investments
        $investments = $user->invests()->where('status', 'ongoing')->get();
        
        if ($investments->count() === 0) {
            $this->warn("⚠️  No active investments found for this user");
            return Command::SUCCESS;
        }
        
        $this->info("📊 Found {$investments->count()} active investments");
        
        $updatedCount = 0;
        
        foreach ($investments as $investment) {
            // Update schema ROI period to 1 minute
            $schema = $investment->schema;
            if ($schema) {
                $schema->update(['roi_period' => $minutes]);
                $this->line("✅ Updated schema '{$schema->name}' ROI period to {$minutes} minute(s)");
            }
            
            // Update existing ROI schedules
            $roiSchedules = RoiSchedule::where('invest_id', $investment->id)
                ->where('status', 'pending')
                ->get();
                
            foreach ($roiSchedules as $schedule) {
                $schedule->update([
                    'roi_start_time' => now(),
                    'roi_end_time' => now()->addMinutes($minutes)
                ]);
                $updatedCount++;
            }
        }
        
        $this->info("✅ Updated {$updatedCount} ROI schedules");
        $this->info("🎯 ROI period set to {$minutes} minute(s) for user: {$username}");
        $this->info("⏰ Next ROI will be ready in {$minutes} minute(s)");
        
        return Command::SUCCESS;
    }
}