<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Invest;
use App\Models\Schema;
use App\Enums\InvestStatus;

class TestInvestmentCancel extends Command
{
    protected $signature = 'test:investment-cancel {username}';
    protected $description = 'Test investment cancellation for a user';

    public function handle()
    {
        $username = $this->argument('username');
        
        $user = User::where('username', $username)->first();
        
        if (!$user) {
            $this->error("User '{$username}' not found!");
            return Command::FAILURE;
        }
        
        $this->info("Testing investment cancellation for user: {$user->username}");
        
        // Get user's ongoing investments
        $investments = Invest::where('user_id', $user->id)
            ->where('status', 'ongoing')
            ->with('schema')
            ->get();
            
        if ($investments->count() === 0) {
            $this->warn("No ongoing investments found for this user");
            return Command::SUCCESS;
        }
        
        $this->info("Found {$investments->count()} ongoing investments:");
        
        foreach ($investments as $investment) {
            $this->line("Investment ID: {$investment->id}");
            $this->line("  Amount: {$investment->invest_amount}");
            $this->line("  Status: {$investment->status}");
            $this->line("  Schema: " . ($investment->schema ? $investment->schema->name : 'No schema'));
            $this->line("  Schema Cancel: " . ($investment->schema ? ($investment->schema->schema_cancel ? 'Yes' : 'No') : 'N/A'));
            $this->line("  Money Back Period: " . ($investment->schema ? ($investment->schema->money_back_period ?? 'N/A') : 'N/A'));
            
            // Check if cancellation fields exist
            $this->line("  Has is_cancelled field: " . (Schema::hasColumn('invests', 'is_cancelled') ? 'Yes' : 'No'));
            $this->line("  Has cancelled_at field: " . (Schema::hasColumn('invests', 'cancelled_at') ? 'Yes' : 'No'));
            $this->line("  Has cancellation_reason field: " . (Schema::hasColumn('invests', 'cancellation_reason') ? 'Yes' : 'No'));
            $this->line("  Has cancellation_deduction field: " . (Schema::hasColumn('invests', 'cancellation_deduction') ? 'Yes' : 'No'));
            $this->line("  Has refund_amount field: " . (Schema::hasColumn('invests', 'refund_amount') ? 'Yes' : 'No'));
            $this->line("");
        }
        
        return Command::SUCCESS;
    }
}
