<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invest;
use App\Models\Schema;
use Illuminate\Support\Facades\DB;

class UpdateInvestmentsToDynamicROI extends Command
{
    protected $signature = 'investments:update-to-dynamic-roi';
    protected $description = 'Update all existing investments to use dynamic ROI from schema';

    public function handle()
    {
        $this->info('🔄 Starting to update investments to dynamic ROI system...');

        try {
            DB::beginTransaction();

            $investments = Invest::with('schema')->get();
            $updatedCount = 0;
            $skippedCount = 0;

            foreach ($investments as $investment) {
                if (!$investment->schema) {
                    $this->warn("Investment ID {$investment->id} has no schema, skipping...");
                    $skippedCount++;
                    continue;
                }

                // Update investment with current schema ROI
                $investment->update([
                    'interest' => $investment->schema->interest,
                    'interest_type' => $investment->schema->interest_type,
                ]);

                $this->info("✅ Updated Investment ID {$investment->id}: ROI {$investment->schema->interest}" . ($investment->schema->interest_type == 'percentage' ? '%' : ''));
                $updatedCount++;
            }

            DB::commit();

            $this->info('=====================================');
            $this->info('🎯 DYNAMIC ROI UPDATE COMPLETED');
            $this->info('=====================================');
            $this->info("✅ Updated: {$updatedCount} investments");
            $this->info("⏭️  Skipped: {$skippedCount} investments");
            $this->info('=====================================');
            $this->info('💡 Now when you change schema ROI, all investments will automatically use the new ROI!');

        } catch (\Exception $e) {
            DB::rollback();
            $this->error('❌ Error updating investments: ' . $e->getMessage());
            return 1;
        }

        return 0;
    }
}
