<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WithdrawMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Validator;

class WithdrawMethodController extends Controller
{
    private function buildRules(): array
    {
        $rules = [
            'name' => 'required|string|max:255',
            'type' => 'required|in:crypto,banking,mobile',
            'icon' => 'nullable|string|max:255',
        ];

        if (Schema::hasColumn('withdraw_methods', 'min_amount')) {
            $rules['min_amount'] = 'required|numeric|min:0';
        }
        if (Schema::hasColumn('withdraw_methods', 'max_amount')) {
            $rules['max_amount'] = 'required|numeric|min:0';
            if (isset($rules['min_amount'])) {
                $rules['max_amount'] .= '|gte:min_amount';
            }
        }
        if (Schema::hasColumn('withdraw_methods', 'fee_percentage')) {
            $rules['fee_percentage'] = 'required|numeric|min:0|max:100';
        }
        if (Schema::hasColumn('withdraw_methods', 'fee_fixed')) {
            $rules['fee_fixed'] = 'required|numeric|min:0';
        }
        if (Schema::hasColumn('withdraw_methods', 'required_fields')) {
            $rules['required_fields'] = 'nullable|array';
        }
        if (Schema::hasColumn('withdraw_methods', 'instructions')) {
            $rules['instructions'] = 'nullable|string';
        }
        if (Schema::hasColumn('withdraw_methods', 'is_active')) {
            $rules['is_active'] = 'boolean';
        }
        if (Schema::hasColumn('withdraw_methods', 'sort_order')) {
            $rules['sort_order'] = 'integer|min:0';
        }

        return $rules;
    }

    private function filterPayload(array $input): array
    {
        $columns = Schema::getColumnListing('withdraw_methods');
        $data = [];
        foreach ($input as $key => $value) {
            if (in_array($key, $columns, true)) {
                $data[$key] = $value;
            }
        }
        return $data;
    }
    public function index()
    {
        $query = WithdrawMethod::query();
        
        // Check if sort_order column exists before ordering by it
        if (Schema::hasColumn('withdraw_methods', 'sort_order')) {
            $query->orderBy('sort_order');
        }
        
        $methods = $query->orderBy('name')->get();
        return view('admin.withdraw.methods.index', compact('methods'));
    }

    public function create()
    {
        return view('admin.withdraw.methods.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), $this->buildRules());

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        $payload = $this->filterPayload($request->all());
        // Provide safe defaults if columns exist but values missing
        if (Schema::hasColumn('withdraw_methods', 'is_active')) {
            $payload['is_active'] = $payload['is_active'] ?? true;
        }
        if (Schema::hasColumn('withdraw_methods', 'sort_order')) {
            $payload['sort_order'] = $payload['sort_order'] ?? 0;
        }
        WithdrawMethod::create($payload);

        return redirect()->route('admin.withdraw.methods.index')
            ->with('success', 'Withdraw method created successfully!');
    }

    public function edit(WithdrawMethod $method)
    {
        return view('admin.withdraw.methods.edit', compact('method'));
    }

    public function update(Request $request, WithdrawMethod $method)
    {
        $validator = Validator::make($request->all(), $this->buildRules());

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        $payload = $this->filterPayload($request->all());
        $method->update($payload);

        return redirect()->route('admin.withdraw.methods.index')
            ->with('success', 'Withdraw method updated successfully!');
    }

    public function destroy(WithdrawMethod $method)
    {
        $method->delete();

        return redirect()->route('admin.withdraw.methods.index')
            ->with('success', 'Withdraw method deleted successfully!');
    }

    public function toggleStatus(WithdrawMethod $method)
    {
        $method->update(['is_active' => !$method->is_active]);

        $status = $method->is_active ? 'activated' : 'deactivated';
        return redirect()->back()
            ->with('success', "Withdraw method {$status} successfully!");
    }
}

