<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\LoginActivities;
use App\Models\Page;
use App\Providers\RouteServiceProvider;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AuthenticatedSessionController extends Controller
{
    public function create()
    {
        $page = Page::where('code', 'login')->where('locale', app()->getLocale())->first();
        $data = json_decode($page->data, true);
        $googleReCaptcha = plugin_active('Google reCaptcha');

        return view('frontend.investflow.auth.login', compact('data', 'googleReCaptcha'));
    }

    public function store(Request $request)
    {
        // Validate the login field (accepts both email and username)
        $request->validate([
            'login' => 'required|string',  // Changed from 'email' to 'login'
            'password' => 'required|string',
        ]);

        // Check login attempts throttle
        $this->ensureIsNotRateLimited($request);

        // Determine if login is email or username
        $field = filter_var($request->login, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

        // Attempt authentication
        if (!Auth::attempt(
            [$field => $request->login, 'password' => $request->password],
            $request->boolean('remember'))
        ) {
            RateLimiter::hit($this->throttleKey($request));
            
            throw ValidationException::withMessages([
                'login' => __('auth.failed'),
            ]);
        }

        RateLimiter::clear($this->throttleKey($request));

        // Successful login
        $oldTheme = session()->get('site-color-mode');
        $request->session()->regenerate();
        LoginActivities::add();
        session()->put('site-color-mode', $oldTheme);

        // Add success notification
        notify()->success('Welcome back! Login successful.');

        return redirect()->intended(RouteServiceProvider::HOME);
    }

    public function destroy(Request $request)
    {
        $oldTheme = session()->get('site-color-mode');
        Auth::guard('web')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        session()->put('site-color-mode', $oldTheme);

        return redirect('/');
    }

    protected function ensureIsNotRateLimited(Request $request): void
    {
        if (!RateLimiter::tooManyAttempts($this->throttleKey($request), 5)) {
            return;
        }

        $seconds = RateLimiter::availableIn($this->throttleKey($request));

        throw ValidationException::withMessages([
            'login' => trans('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    protected function throttleKey(Request $request): string
    {
        return Str::transliterate(Str::lower($request->input('login'))).'|'.$request->ip();
    }
}