<?php

namespace App\Http\Controllers\Auth;

use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Events\UserReferred;
use App\Http\Controllers\Controller;
use App\Models\LoginActivities;
use App\Models\Page;
use App\Models\Ranking;
use App\Models\User;
use App\Providers\RouteServiceProvider;
use App\Rules\Recaptcha;
use App\Traits\NotifyTrait;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;
use Session;
use Txn;

class RegisteredUserController extends Controller
{
    use NotifyTrait;

    /**
     * Handle an incoming registration request.
     *
     * @return RedirectResponse
     *
     * @throws ValidationException
     */
    public function store(Request $request)
    {
        // FAST REGISTRATION - Cached settings
        static $pageSettings = null;
        if ($pageSettings === null) {
            $pageSettings = \App\Models\PageSetting::pluck('value', 'key')->toArray();
        }
        
        $isUsername = (bool)($pageSettings['username_show'] ?? false);
        $isCountry = (bool)($pageSettings['country_show'] ?? false);
        $isPhone = (bool)($pageSettings['phone_show'] ?? false);
        try {
            $request->validate([
                'first_name' => ['required', 'string', 'max:255'],
                'last_name' => ['required', 'string', 'max:255'],
                'username' => $isUsername ? ['required', 'string', 'max:255', 'unique:users'] : ['nullable'],
                'country' => $isCountry ? ['required', 'string', 'max:255'] : ['nullable'],
                'phone' => $isPhone ? ['required', 'string', 'max:255'] : ['nullable'],
                'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
                'password' => ['required', 'confirmed'],
                'i_agree' => ['required'],
            ]);
        } catch (\Exception $e) {
            throw $e;
        }

        $input = $request->all();

        $location = getLocation();
        $phone = $isPhone ? $location->dial_code . ' ' . $input['phone'] : $location->dial_code . ' ';
        $country = $isCountry ? $input['country'] : $location->name;

        // New users start with NO RANK as default
        static $defaultRank = null;
        if ($defaultRank === null) {
            $defaultRank = Ranking::where('ranking_name', 'NO RANK')->where('ranking', 0)->first(); // NO RANK as default starting rank
        }

        $user = User::create([
            'ranking_id' => $defaultRank ? $defaultRank->id : null,
            'rankings' => json_encode([]), // Empty array - no ranks achieved yet
            'first_name' => $input['first_name'],
            'last_name' => $input['last_name'],
            'username' => $isUsername ? $input['username'] : $input['first_name'] . $input['last_name'] . rand(1000, 9999),
            'country' => $country,
            'phone' => $phone,
            'email' => $input['email'],
            'password' => Hash::make($input['password']),
            // ✅ ADD REFERRAL SYSTEM
            'ref_id' => $this->getReferralId($request),
        ]);

        // FAST REGISTRATION - Login immediately
        try {
            \Cookie::forget('invite');
            Auth::login($user);
            \Log::info('User logged in successfully', ['user_id' => $user->id, 'email' => $user->email]);
        } catch (\Exception $e) {
            \Log::error('Login failed after registration', ['error' => $e->getMessage(), 'user_id' => $user->id]);
            // Continue anyway
        }

        \Log::info('Redirecting to dashboard', ['user_id' => $user->id, 'is_authenticated' => Auth::check()]);
        return redirect(RouteServiceProvider::HOME);
    }

    /**
     * Get referral ID from request or cookie
     */
    private function getReferralId(Request $request)
    {
        // ✅ ADD DEBUG LOGGING
        \Log::info('🔍 getReferralId() called', [
            'has_invite_cookie' => $request->hasCookie('invite'),
            'invite_cookie_value' => $request->cookie('invite'),
            'has_invite_field' => $request->has('invite'),
            'invite_field_value' => $request->input('invite'),
            'has_ref_id' => $request->has('ref_id'),
            'ref_id_value' => $request->input('ref_id'),
        ]);
        
        // ✅ PRIORITY 1: Check for invite field in form (most common)
        if ($request->has('invite') && !empty($request->input('invite'))) {
            $inviteCode = trim($request->input('invite'));
            \Log::info('🔍 Found invite field', ['value' => $inviteCode]);
            
            // Find referral link by code
            $referralLink = \App\Models\ReferralLink::where('code', $inviteCode)->first();
            if ($referralLink) {
                \Log::info('🔍 Found referral link by invite code', [
                    'code' => $inviteCode,
                    'user_id' => $referralLink->user_id
                ]);
                return $referralLink->user_id;
            } else {
                \Log::warning('⚠️ Referral link not found for invite code', ['code' => $inviteCode]);
            }
        }
        
        // ✅ PRIORITY 2: Check for invite cookie
        $inviteCookie = $request->cookie('invite');
        if ($inviteCookie) {
            \Log::info('🔍 Found invite cookie', ['value' => $inviteCookie]);
            
            // Find referral link by code
            $referralLink = \App\Models\ReferralLink::where('code', $inviteCookie)->first();
            if ($referralLink) {
                \Log::info('🔍 Found referral link by cookie', [
                    'code' => $inviteCookie,
                    'user_id' => $referralLink->user_id
                ]);
                return $referralLink->user_id;
            } else {
                \Log::warning('⚠️ Referral link not found for cookie code', ['code' => $inviteCookie]);
            }
        }
        
        // ✅ PRIORITY 3: Check for ref_id in request (fallback)
        if ($request->has('ref_id') && !empty($request->input('ref_id'))) {
            $refId = $request->input('ref_id');
            \Log::info('🔍 Found ref_id in request', ['value' => $refId]);
            
            // Verify user exists
            if (\App\Models\User::find($refId)) {
                \Log::info('🔍 User exists, returning ref_id', ['ref_id' => $refId]);
                return $refId;
            } else {
                \Log::warning('⚠️ User not found for ref_id', ['value' => $refId]);
            }
        }
        
        \Log::info('🔍 No referral found, returning null');
        return null;
    }

    /**
     * Display the registration view.
     *
     * @return View
     */
    public function create()
    {
        // FAST PAGE LOAD - Cached settings
        static $pageSettings = null;
        if ($pageSettings === null) {
            $pageSettings = \App\Models\PageSetting::pluck('value', 'key')->toArray();
        }
        
        if (!($pageSettings['account_creation'] ?? true)) {
            abort('403', 'User registration is closed now');
        }

        // Cache page data
        static $pageData = null;
        if ($pageData === null) {
            $page = Page::where('code', 'registration')->where('locale', app()->getLocale())->first();
            $pageData = json_decode($page->data ?? '{}', true);
        }

        // Cache plugin status
        static $googleReCaptcha = null;
        if ($googleReCaptcha === null) {
            $googleReCaptcha = plugin_active('Google reCaptcha');
        }
        
        // Cache location
        $location = getLocation();

        return view('frontend.investflow.auth.register', compact('location', 'googleReCaptcha', 'pageData'));
    }
}
