<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Ranking;
use App\Traits\ImageUpload;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Validator;

class RankingController extends Controller
{
    use ImageUpload;

    public function __construct()
    {
        $this->middleware('permission:ranking-list|ranking-create|ranking-edit', ['only' => ['index', 'store']]);
        $this->middleware('permission:ranking-create', ['only' => ['store']]);
        $this->middleware('permission:ranking-edit', ['only' => ['update']]);
        $this->middleware('permission:ranking-delete', ['only' => ['delete']]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return Application|Factory|View
     */
    public function index()
    {
        $rankings = Ranking::all();
        return view('backend.ranking.index', compact('rankings'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return RedirectResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'icon' => 'sometimes|nullable|image|mimes:jpg,png,svg',
            'ranking' => 'required',
            'ranking_name' => 'required',
            'minimum_earnings' => 'nullable|numeric|min:0',
            'minimum_deposit' => 'nullable|numeric|min:0',
            'minimum_invest' => 'nullable|numeric|min:0',
            'minimum_referral' => 'nullable|numeric|min:0',
            'minimum_referral_deposit' => 'nullable|numeric|min:0',
            'minimum_referral_invest' => 'nullable|numeric|min:0',
            'bonus' => 'nullable|numeric|min:0',
            'salary' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'status' => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        $input = $request->all();

        Ranking::create([
            'icon' => $request->hasFile('icon') ? self::imageUploadTrait($input['icon']) : 'global/materials/upload.svg',
            'ranking' => $input['ranking'],
            'ranking_name' => $input['ranking_name'],
            'minimum_earnings' => $input['minimum_earnings'] ?? 0,
            'minimum_deposit' => $input['minimum_deposit'] ?? 0,
            'minimum_invest' => $input['minimum_invest'] ?? 0,
            'minimum_referral' => $input['minimum_referral'] ?? 0,
            'minimum_referral_deposit' => $input['minimum_referral_deposit'] ?? 0,
            'minimum_referral_invest' => $input['minimum_referral_invest'] ?? 0,
            'bonus' => $input['bonus'] ?? 0,
            'salary' => $input['salary'] ?? 0,
            'description' => $input['description'] ?? null,
            'status' => $input['status'],
        ]);

        notify()->success('Ranking created successfully');
        return redirect()->route('admin.ranking.index');
    }

    /**
     * Update the specified resource in storage.
     *
     * @return RedirectResponse
     */
    public function update(Request $request, Ranking $ranking)
    {
        $validator = Validator::make($request->all(), [
            'ranking' => 'required',
            'ranking_name' => 'required',
            'minimum_earnings' => 'nullable|numeric|min:0',
            'minimum_deposit' => 'nullable|numeric|min:0',
            'minimum_invest' => 'nullable|numeric|min:0',
            'minimum_referral' => 'nullable|numeric|min:0',
            'minimum_referral_deposit' => 'nullable|numeric|min:0',
            'minimum_referral_invest' => 'nullable|numeric|min:0',
            'bonus' => 'nullable|numeric|min:0',
            'salary' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'status' => 'required|in:0,1',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        if ($ranking->id == 1 && $request->status == 0) {
            notify()->error('Default Rank Status Not Updated', 'Error');
            return redirect()->back();
        }

        $data = [
            'ranking' => $request->ranking,
            'ranking_name' => $request->ranking_name,
            'minimum_earnings' => $request->minimum_earnings ?? 0,
            'minimum_deposit' => $request->minimum_deposit ?? 0,
            'minimum_invest' => $request->minimum_invest ?? 0,
            'minimum_referral' => $request->minimum_referral ?? 0,
            'minimum_referral_deposit' => $request->minimum_referral_deposit ?? 0,
            'minimum_referral_invest' => $request->minimum_referral_invest ?? 0,
            'bonus' => $request->bonus ?? 0,
            'salary' => $request->salary ?? 0,
            'description' => $request->description ?? null,
            'status' => $request->status,
        ];

        if ($request->hasFile('icon')) {
            $icon = self::imageUploadTrait($request->icon, $ranking->icon);
            $data['icon'] = $icon;
        }

        $ranking->update($data);

        notify()->success('Ranking updated successfully');
        return redirect()->route('admin.ranking.index');
    }
}
