<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class ReferralManagementController extends Controller
{
    public function index()
    {
        $users = User::select('id', 'username', 'email', 'ref_id', 'created_at')
            ->with('referrer:id,username')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        $totalUsers = User::count();
        $usersWithReferrers = User::whereNotNull('ref_id')->count();
        $usersWithoutReferrers = User::whereNull('ref_id')->count();

        return view('backend.referral-management.index', compact(
            'users',
            'totalUsers',
            'usersWithReferrers',
            'usersWithoutReferrers'
        ));
    }

    public function generateAvatarForUser($userId)
    {
        try {
            $user = User::find($userId);
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ]);
            }

            $avatarPath = $this->generateAvatar($user->username, $user->id);
            
            // Update user's avatar
            $user->avatar = $avatarPath;
            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'Avatar generated successfully!',
                'data' => [
                    'user' => $user->username,
                    'avatar' => $avatarPath
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error generating avatar: ' . $e->getMessage()
            ]);
        }
    }

    public function generateAvatarsForAllUsers()
    {
        try {
            $users = User::all();
            $generatedCount = 0;
            $errors = [];

            foreach ($users as $user) {
                try {
                    $avatarPath = $this->generateAvatar($user->username, $user->id);
                    $user->avatar = $avatarPath;
                    $user->save();
                    $generatedCount++;
                } catch (\Exception $e) {
                    $errors[] = "User {$user->username}: " . $e->getMessage();
                }
            }

            return response()->json([
                'success' => true,
                'message' => "Generated avatars for {$generatedCount} users successfully!",
                'data' => [
                    'generated_count' => $generatedCount,
                    'total_users' => $users->count(),
                    'errors' => $errors
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error generating avatars: ' . $e->getMessage()
            ]);
        }
    }

    private function generateAvatar($username, $userId)
    {
        // Create avatar directory if it doesn't exist
        $avatarDir = 'public/avatars';
        if (!Storage::exists($avatarDir)) {
            Storage::makeDirectory($avatarDir);
        }

        // Generate unique filename
        $filename = 'avatar_' . $userId . '_' . time() . '.svg';
        $filepath = $avatarDir . '/' . $filename;

        // Generate SVG avatar
        $svg = $this->generateSVGAvatar($username);
        
        // Save SVG file
        Storage::put($filepath, $svg);

        // Return the public URL - fix the path to match storage link
        return '/storage/avatars/' . $filename;
    }

    private function generateSVGAvatar($username)
    {
        // Get first letter of username
        $initial = strtoupper(substr($username, 0, 1));
        
        // Generate random colors
        $colors = [
            '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', '#FFEAA7',
            '#DDA0DD', '#98D8C8', '#F7DC6F', '#BB8FCE', '#85C1E9',
            '#F8C471', '#82E0AA', '#F1948A', '#85C1E9', '#D7BDE2'
        ];
        $bgColor = $colors[array_rand($colors)];
        
        // Generate complementary text color
        $textColor = $this->getContrastColor($bgColor);

        $svg = '<?xml version="1.0" encoding="UTF-8"?>
        <svg width="100" height="100" viewBox="0 0 100 100" xmlns="http://www.w3.org/2000/svg">
            <defs>
                <linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="100%">
                    <stop offset="0%" style="stop-color:' . $bgColor . ';stop-opacity:1" />
                    <stop offset="100%" style="stop-color:' . $this->darkenColor($bgColor, 20) . ';stop-opacity:1" />
                </linearGradient>
            </defs>
            <circle cx="50" cy="50" r="50" fill="url(#grad)" />
            <text x="50" y="65" font-family="Arial, sans-serif" font-size="40" font-weight="bold" text-anchor="middle" fill="' . $textColor . '">' . $initial . '</text>
        </svg>';

        return $svg;
    }

    private function getContrastColor($hexColor)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Calculate luminance
        $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
        
        // Return black or white based on luminance
        return $luminance > 0.5 ? '#000000' : '#FFFFFF';
    }

    private function darkenColor($hexColor, $percent)
    {
        // Remove # if present
        $hexColor = ltrim($hexColor, '#');
        
        // Convert to RGB
        $r = hexdec(substr($hexColor, 0, 2));
        $g = hexdec(substr($hexColor, 2, 2));
        $b = hexdec(substr($hexColor, 4, 2));
        
        // Darken by percentage
        $r = max(0, min(255, $r - ($r * $percent / 100)));
        $g = max(0, min(255, $g - ($g * $percent / 100)));
        $b = max(0, min(255, $b - ($b * $percent / 100)));
        
        // Convert back to hex
        return sprintf("#%02x%02x%02x", $r, $g, $b);
    }

    public function assignReferrer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'referrer_id' => 'required|exists:users,id|different:user_id',
        ], [
            'user_id.required' => 'User is required',
            'referrer_id.required' => 'Referrer is required',
            'referrer_id.different' => 'User cannot refer themselves',
            'user_id.exists' => 'Selected user does not exist',
            'referrer_id.exists' => 'Selected referrer does not exist',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ]);
        }

        $userId = $request->user_id;
        $referrerId = $request->referrer_id;

        // Check if user already has a referrer
        $user = User::find($userId);
        if ($user->ref_id) {
            return response()->json([
                'success' => false,
                'message' => 'User already has a referrer: ' . User::find($user->ref_id)->username
            ]);
        }

        // Check for circular references
        if ($this->wouldCreateCircularReference($userId, $referrerId)) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot create circular reference in referral chain'
            ]);
        }

        try {
            DB::beginTransaction();

            // Update user's ref_id
            $user->ref_id = $referrerId;
            $user->save();

            // Create referral relationship record if table exists
            if (DB::getSchemaBuilder()->hasTable('referral_relationships')) {
                // Get or create referral link for the referrer
                $referralLink = DB::table('referral_links')
                    ->where('user_id', $referrerId)
                    ->first();
                
                if (!$referralLink) {
                    // Create referral link if it doesn't exist
                    $referralLinkId = DB::table('referral_links')->insertGetId([
                        'user_id' => $referrerId,
                        'referral_program_id' => 1, // Default referral program
                        'code' => User::find($referrerId)->username,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                } else {
                    $referralLinkId = $referralLink->id;
                }
                
                // Insert into referral_relationships with correct structure
                DB::table('referral_relationships')->insert([
                    'referral_link_id' => $referralLinkId,
                    'user_id' => $userId,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Referrer assigned successfully!',
                'data' => [
                    'user' => $user->username,
                    'referrer' => User::find($referrerId)->username
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error assigning referrer: ' . $e->getMessage()
            ]);
        }
    }

    public function removeReferrer(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ]);
        }

        try {
            DB::beginTransaction();

            $user = User::find($request->user_id);
            $oldReferrerId = $user->ref_id;
            
            if (!$oldReferrerId) {
                return response()->json([
                    'success' => false,
                    'message' => 'User has no referrer to remove'
                ]);
            }

            $oldReferrer = User::find($oldReferrerId);
            
            // Remove referrer
            $user->ref_id = null;
            $user->save();

            // Remove referral relationship record if table exists
            if (DB::getSchemaBuilder()->hasTable('referral_relationships')) {
                DB::table('referral_relationships')
                    ->where('user_id', $request->user_id)
                    ->delete();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Referrer removed successfully!',
                'data' => [
                    'user' => $user->username,
                    'old_referrer' => $oldReferrer->username
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Error removing referrer: ' . $e->getMessage()
            ]);
        }
    }

    public function getReferralTree($userId = null)
    {
        if (!$userId) {
            $userId = User::first()->id;
        }

        $user = User::find($userId);
        if (!$user) {
            return response()->json(['success' => false, 'message' => 'User not found']);
        }

        $referrals = User::where('ref_id', $userId)
            ->select('id', 'username', 'email', 'created_at')
            ->withCount('referrals')
            ->get();

        $tree = [
            'id' => $user->id,
            'username' => $user->username,
            'email' => $user->email,
            'referrals' => $referrals->map(function ($referral) {
                return [
                    'id' => $referral->id,
                    'username' => $referral->username,
                    'email' => $referral->email,
                    'referral_count' => $referral->referrals_count,
                    'join_date' => $referral->created_at->format('M d, Y')
                ];
            })
        ];

        return response()->json([
            'success' => true,
            'data' => $tree
        ]);
    }

    private function wouldCreateCircularReference($userId, $referrerId)
    {
        // Check if the referrer is already a referral of the user
        $currentReferrer = $referrerId;
        
        while ($currentReferrer) {
            if ($currentReferrer == $userId) {
                return true; // Circular reference detected
            }
            
            $user = User::find($currentReferrer);
            $currentReferrer = $user ? $user->ref_id : null;
        }
        
        return false;
    }

    public function searchUsers(Request $request)
    {
        $query = $request->get('q', '');
        $limit = $request->get('limit', 10);
        
        $userQuery = User::select('id', 'username', 'email');
        
        if ($query) {
            $userQuery->where(function($q) use ($query) {
                $q->where('username', 'like', "%{$query}%")
                  ->orWhere('email', 'like', "%{$query}%");
            });
        }
        
        $users = $userQuery->limit($limit)->get();

        return response()->json($users);
    }
}
