<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Invest;
use App\Models\Transaction;
use App\Models\RoiDistribution;
use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class RoiTrackingController extends Controller
{
    public function index(Request $request)
    {
        $query = User::with(['invests' => function($q) {
            $q->where('status', InvestStatus::Ongoing);
        }, 'transactions' => function($q) {
            $q->where('type', 'roi_profit');
        }]);

        // Filter by status
        if ($request->filled('status')) {
            $status = $request->status;
            if ($status === 'with_investment') {
                $query->whereHas('invests', function($q) {
                    $q->where('status', InvestStatus::Ongoing);
                });
            } elseif ($status === 'without_investment') {
                $query->whereDoesntHave('invests', function($q) {
                    $q->where('status', InvestStatus::Ongoing);
                });
            }
        }

        // Filter by date
        if ($request->filled('date')) {
            $date = Carbon::parse($request->date);
            $query->whereHas('transactions', function($q) use ($date) {
                $q->where('type', 'roi_profit')
                  ->whereDate('created_at', $date);
            });
        }

        $users = $query->paginate(20);

        // Get statistics
        $totalUsers = User::count();
        $usersWithInvestment = User::whereHas('invests', function($q) {
            $q->where('status', InvestStatus::Ongoing);
        })->count();
        $usersWithoutInvestment = $totalUsers - $usersWithInvestment;

        // Get today's ROI count
        $todayRoiCount = Transaction::where('type', 'roi_profit')
            ->whereDate('created_at', Carbon::today())
            ->count();

        // Get today's ROI data for statistics
        $todayRoiData = $this->getTodayRoiData();

        // Get recent ROI distributions (with error handling for missing table)
        $recentDistributions = collect();
        try {
            $recentDistributions = RoiDistribution::with(['user', 'investment.schema'])
                ->orderBy('distribution_time', 'desc')
                ->limit(10)
                ->get();
        } catch (\Exception $e) {
            // Table doesn't exist yet, use empty collection
            $recentDistributions = collect();
        }

        return view('backend.roi-tracking.index', compact(
            'users', 
            'totalUsers', 
            'usersWithInvestment', 
            'usersWithoutInvestment',
            'todayRoiCount',
            'todayRoiData',
            'recentDistributions'
        ));
    }

    public function userDetail($userId)
    {
        $user = User::with(['invests' => function($q) {
            $q->where('status', InvestStatus::Ongoing);
        }, 'transactions' => function($q) {
            $q->where('type', 'roi_profit');
        }])->findOrFail($userId);

        return view('backend.roi-tracking.user-detail', compact('user'));
    }

    /**
     * Get today's ROI statistics
     */
    private function getTodayRoiData()
    {
        $today = Carbon::today();
        $tomorrow = Carbon::tomorrow();

        // Get investments due today
        $investmentsDueToday = Invest::with(['user', 'schema'])
            ->where('status', InvestStatus::Ongoing)
            ->where('next_profit_time', '>=', $today)
            ->where('next_profit_time', '<', $tomorrow)
            ->get();

        $dueToday = $investmentsDueToday->count();
        $totalRoiToday = 0;

        foreach ($investmentsDueToday as $investment) {
            $roiAmount = $this->calculateRoi($investment);
            $totalRoiToday += $roiAmount;
        }

        // Get processed ROI today
        $processedToday = Transaction::where('type', 'roi_profit')
            ->whereDate('created_at', $today)
            ->count();

        return [
            'due_today' => $dueToday,
            'processed_today' => $processedToday,
            'total_roi_today' => $totalRoiToday,
            'investments_due' => $investmentsDueToday
        ];
    }

    /**
     * Bulk distribute ROI to all users with active investments
     */
    public function bulkDistributeRoi(Request $request)
    {
        try {
            DB::beginTransaction();

            $distributedCount = 0;
            $totalRoiAmount = 0;
            $errors = [];

            // Get all active investments
            $activeInvestments = Invest::with(['user', 'schema'])
                ->where('status', InvestStatus::Ongoing)
                ->get();

            foreach ($activeInvestments as $investment) {
                try {
                    $roiAmount = $this->calculateRoi($investment);
                    
                    if ($roiAmount > 0) {
                        // Create ROI transaction
                        $transaction = Transaction::create([
                            'user_id' => $investment->user_id,
                            'type' => 'roi_profit',
                            'amount' => $roiAmount,
                            'status' => TxnStatus::Completed,
                            'description' => 'Bulk ROI from investment plan: ' . ($investment->schema->name ?? 'Unknown Plan'),
                            'tnx' => 'ROI' . strtoupper(uniqid()),
                            'created_at' => Carbon::now(),
                            'updated_at' => Carbon::now()
                        ]);

                        // Update user balance
                        $investment->user->increment('balance', $roiAmount);

                        // Update investment progress
                        $investment->increment('already_return_profit');
                        
                        // Align next cycle to fixed anchor from investment start (not collection time)
                        // Normalize to minutes
                        $rawTime = optional($investment->schema->schedule)->time ?? ($investment->period_hours ?: 24);
                        $scheduleTime = $rawTime > 48 ? $rawTime : ($rawTime * 60);
                        $anchorStart = Carbon::parse($investment->created_at);
                        $elapsedMinutes = $anchorStart->diffInMinutes(Carbon::now());
                        $cyclesElapsed = intdiv(max(0, $elapsedMinutes), max(1, $scheduleTime));
                        $nextWindowStart = $anchorStart->copy()->addMinutes(($cyclesElapsed + 1) * $scheduleTime);

                        $investment->update([
                            'last_profit_time' => Carbon::now(),
                            'next_profit_time' => $nextWindowStart,
                            'total_profit_amount' => ($investment->total_profit_amount ?? 0) + $roiAmount
                        ]);

                        // Check if investment should be completed
                        if ($investment->return_type === 'period' && 
                            $investment->already_return_profit >= $investment->number_of_period) {
                            $investment->update(['status' => InvestStatus::Completed]);
                            
                            // Return capital if enabled
                            if ($investment->capital_back) {
                                $investment->user->increment('balance', $investment->invest_amount);
                                
                                // Create capital return transaction
                                Transaction::create([
                                    'user_id' => $investment->user_id,
                                    'type' => 'capital_return',
                                    'amount' => $investment->invest_amount,
                                    'status' => TxnStatus::Completed,
                                    'description' => 'Capital return from completed investment: ' . ($investment->schema->name ?? 'Unknown Plan'),
                                    'tnx' => 'CAP' . strtoupper(uniqid()),
                                    'created_at' => Carbon::now(),
                                    'updated_at' => Carbon::now()
                                ]);
                            }
                        }

                        // Record ROI distribution (with error handling for missing table)
                        try {
                            RoiDistribution::create([
                                'user_id' => $investment->user_id,
                                'investment_id' => $investment->id,
                                'roi_amount' => $roiAmount,
                                'distribution_type' => 'bulk',
                                'distributed_by' => auth()->id(),
                                'distribution_time' => Carbon::now(),
                                'notes' => 'Bulk ROI distribution'
                            ]);
                        } catch (\Exception $e) {
                            // Table doesn't exist yet, skip recording
                            // This allows ROI distribution to work even without the tracking table
                        }

                        $distributedCount++;
                        $totalRoiAmount += $roiAmount;
                    }
                } catch (\Exception $e) {
                    $errors[] = "Investment ID {$investment->id}: " . $e->getMessage();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Bulk ROI distribution completed! {$distributedCount} investments processed.",
                'data' => [
                    'distributed_count' => $distributedCount,
                    'total_roi_amount' => $totalRoiAmount,
                    'errors' => $errors
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Bulk ROI distribution failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reverse ROI distribution
     */
    public function reverseRoiDistribution(Request $request)
    {
        $request->validate([
            'distribution_id' => 'required|exists:roi_distributions,id',
            'reversal_reason' => 'required|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            $distribution = RoiDistribution::with(['user', 'investment'])->findOrFail($request->distribution_id);

            // Check if can be reversed
            if (!$distribution->canBeReversed()) {
                return response()->json([
                    'success' => false,
                    'message' => 'This ROI distribution cannot be reversed. It may have already been reversed or the 24-hour window has expired.'
                ], 400);
            }

            // Reverse the ROI
            $user = $distribution->user;
            $roiAmount = $distribution->roi_amount;

            // Deduct from user balance
            $user->decrement('balance', $roiAmount);

            // Revert investment progress
            $investment = $distribution->investment;
            $investment->decrement('already_return_profit');
            $investment->update([
                'total_profit_amount' => max(0, ($investment->total_profit_amount ?? 0) - $roiAmount)
            ]);

            // Mark transaction as reversed
            $transaction = Transaction::where('user_id', $user->id)
                ->where('type', 'roi_profit')
                ->where('amount', $roiAmount)
                ->whereDate('created_at', $distribution->distribution_time)
                ->first();

            if ($transaction) {
                $transaction->update(['status' => TxnStatus::Failed]);
            }

            // Update distribution record
            $distribution->update([
                'is_reversed' => true,
                'reversed_at' => Carbon::now(),
                'reversed_by' => auth()->id(),
                'reversal_reason' => $request->reversal_reason
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'ROI distribution reversed successfully!',
                'data' => [
                    'user' => $user->full_name,
                    'amount' => $roiAmount,
                    'reversal_reason' => $request->reversal_reason
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Failed to reverse ROI distribution: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get ROI distribution history
     */
    public function distributionHistory(Request $request)
    {
        $query = RoiDistribution::with(['user', 'investment.schema', 'distributedBy']);

        // Filter by date range
        if ($request->filled('start_date')) {
            $query->whereDate('distribution_time', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->whereDate('distribution_time', '<=', $request->end_date);
        }

        // Filter by status
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->active();
            } elseif ($request->status === 'reversed') {
                $query->reversed();
            }
        }

        $distributions = $query->orderBy('distribution_time', 'desc')->paginate(50);

        return view('backend.roi-tracking.distribution-history', compact('distributions'));
    }

    public function processRoi(Request $request)
    {
        try {
            $users = User::whereHas('invests', function($q) {
                $q->where('status', InvestStatus::Ongoing);
            })->get();

            $processedCount = 0;
            foreach ($users as $user) {
                foreach ($user->invests as $investment) {
                    if ($investment->status === InvestStatus::Ongoing) {
                        // Calculate ROI based on investment plan
                        $roiAmount = $this->calculateRoi($investment);
                        
                        if ($roiAmount > 0) {
                            // Create ROI transaction
                            Transaction::create([
                                'user_id' => $user->id,
                                'type' => 'roi_profit',
                                'amount' => $roiAmount,
                                'status' => TxnStatus::Completed,
                                'description' => 'Daily ROI from investment plan: ' . $investment->plan_name
                            ]);

                            // Update user balance
                            $user->increment('balance', $roiAmount);
                            $processedCount++;
                        }
                    }
                }
            }

            notify()->success("ROI processed for {$processedCount} investments successfully!", 'Success');
        } catch (\Exception $e) {
            notify()->error('Error processing ROI: ' . $e->getMessage(), 'Error');
        }

        return redirect()->back();
    }

    private function calculateRoi($investment)
    {
        try {
            // Get the schema for this investment
            $schema = $investment->schema;
            
            if (!$schema) {
                return 0;
            }
            
            // Calculate ROI based on schema settings
            $roiAmount = 0;
            
            if ($schema->interest_type === 'percentage') {
                // Percentage-based ROI
                $roiAmount = ($schema->return_interest * $investment->invest_amount) / 100;
            } else {
                // Fixed amount ROI
                $roiAmount = $schema->return_interest;
            }
            
            // Check if this is a daily, hourly, or other frequency
            $scheduleMinutes = $schema->schedule->time ?? 1440; // Default to 24 hours (1440 minutes)
            
            // For different frequency plans, adjust ROI accordingly
            if ($scheduleMinutes < 1440) { // Less than 24 hours
                $roiAmount = ($roiAmount * $scheduleMinutes) / 1440;
            }
            
            return round($roiAmount, 2);
            
        } catch (\Exception $e) {
            // Fallback to basic calculation if schema is not available
            if (isset($investment->interest) && isset($investment->invest_amount)) {
                return round(($investment->interest * $investment->invest_amount) / 100, 2);
            }
            
            return 0;
        }
    }

    public function debug()
    {
        $data = [
            'total_users' => User::count(),
            'users_with_investment' => User::whereHas('invests', function($q) {
                $q->where('status', InvestStatus::Ongoing);
            })->count(),
            'users_without_investment' => User::whereDoesntHave('invests', function($q) {
                $q->where('status', InvestStatus::Ongoing);
            })->count(),
            'today_roi_count' => Transaction::where('type', 'roi_profit')
                ->whereDate('created_at', Carbon::today())
                ->count()
        ];

        return response()->json($data);
    }

    /**
     * Get user's today ROI for API
     */
    public function getUserTodayRoiApi($userId)
    {
        $user = User::with(['invests' => function($q) {
            $q->where('status', InvestStatus::Ongoing);
        }])->findOrFail($userId);

        $today = Carbon::today();
        $tomorrow = Carbon::tomorrow();

        $investments = $user->invests->where('next_profit_time', '>=', $today)
            ->where('next_profit_time', '<', $tomorrow);

        $roiData = [];
        $totalRoi = 0;

        foreach ($investments as $investment) {
            $roiAmount = $this->calculateRoi($investment);
            $roiData[] = [
                'invest' => $investment,
                'roi_amount' => $roiAmount,
                'schema' => $investment->schema
            ];
            $totalRoi += $roiAmount;
        }

        return response()->json([
            'investments' => $roiData,
            'total_roi' => $totalRoi
        ]);
    }

    /**
     * Process single ROI for a user
     */
    public function processSingleRoi(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'investment_id' => 'required|exists:invests,id'
        ]);

        try {
            DB::beginTransaction();

            $investment = Invest::with(['user', 'schema'])->findOrFail($request->investment_id);
            
            if ($investment->user_id != $request->user_id) {
                throw new \Exception('Investment does not belong to the specified user');
            }

            $roiAmount = $this->calculateRoi($investment);
            
            if ($roiAmount > 0) {
                // Create ROI transaction
                $transaction = Transaction::create([
                    'user_id' => $investment->user_id,
                    'type' => 'roi_profit',
                    'amount' => $roiAmount,
                    'status' => TxnStatus::Completed,
                    'description' => 'Manual ROI from investment plan: ' . ($investment->schema->name ?? 'Unknown Plan'),
                    'tnx' => 'ROI' . strtoupper(uniqid()),
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);

                // Update user balance
                $investment->user->increment('balance', $roiAmount);

                // Update investment progress
                $investment->increment('already_return_profit');
                // Calculate next profit time based on fixed anchor (investment created_at)
                $rawTime = optional($investment->schema->schedule)->time ?? ($investment->period_hours ?: 24);
                $scheduleMinutes = $rawTime > 48 ? $rawTime : ($rawTime * 60); // Default 24 hours normalized
                $anchorStart = Carbon::parse($investment->created_at);
                $elapsedMinutes = $anchorStart->diffInMinutes(Carbon::now());
                $cyclesElapsed = intdiv(max(0, $elapsedMinutes), max(1, $scheduleMinutes));
                $nextWindowStart = $anchorStart->copy()->addMinutes(($cyclesElapsed + 1) * $scheduleMinutes);
                $investment->update([
                    'last_profit_time' => Carbon::now(),
                    'next_profit_time' => $nextWindowStart,
                    'total_profit_amount' => ($investment->total_profit_amount ?? 0) + $roiAmount
                ]);

                                        // Record ROI distribution (with error handling for missing table)
                        try {
                            RoiDistribution::create([
                                'user_id' => $investment->user_id,
                                'investment_id' => $investment->id,
                                'roi_amount' => $roiAmount,
                                'distribution_type' => 'manual',
                                'distributed_by' => auth()->id(),
                                'distribution_time' => Carbon::now(),
                                'notes' => 'Manual ROI processing'
                            ]);
                        } catch (\Exception $e) {
                            // Table doesn't exist yet, skip recording
                            // This allows ROI processing to work even without the tracking table
                        }

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'ROI processed successfully!',
                    'data' => [
                        'user' => $investment->user->full_name,
                        'amount' => $roiAmount,
                        'next_profit_time' => $investment->next_profit_time
                    ]
                ]);
            } else {
                throw new \Exception('No ROI amount calculated for this investment');
            }

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Failed to process ROI: ' . $e->getMessage()
            ], 500);
        }
    }
}
