<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Slider;
use App\Traits\ImageUpload;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class SliderController extends Controller
{
    use ImageUpload;

    public function index(): View
    {
        $sliders = collect();
        try {
            if (\Illuminate\Support\Facades\Schema::hasTable('sliders')) {
                $sliders = Slider::orderBy('position')->get();
            }
        } catch (\Throwable $e) {
            $sliders = collect();
        }
        return view('backend.page.slider.index', compact('sliders'));
    }

    public function store(Request $request): RedirectResponse
    {
        if (!\Illuminate\Support\Facades\Schema::hasTable('sliders')) {
            notify()->error(__('Please run migrations to create sliders table first.'));
            return back();
        }
        $request->validate([
            'title' => 'required|string|max:255',
            'image' => 'required|file|mimes:jpg,jpeg,png,gif,webp',
            'desktop_image' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp',
            'mobile_image' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp',
            'link' => 'nullable|url',
            'position' => 'nullable|integer',
            'status' => 'nullable|boolean',
        ]);

        $data = [
            'title' => $request->title,
            'image' => self::imageUploadTrait($request->file('image')),
            'link' => $request->link,
            'position' => $request->position ?? 0,
            'status' => (bool) ($request->status ?? true),
        ];

        if ($request->hasFile('desktop_image')) {
            $data['desktop_image'] = self::imageUploadTrait($request->file('desktop_image'));
        }
        if ($request->hasFile('mobile_image')) {
            $data['mobile_image'] = self::imageUploadTrait($request->file('mobile_image'));
        }

        Slider::create($data);
        notify()->success(__('Slider added'));
        return back();
    }

    public function update(Request $request, Slider $slider): RedirectResponse
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'image' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp',
            'desktop_image' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp',
            'mobile_image' => 'nullable|file|mimes:jpg,jpeg,png,gif,webp',
            'link' => 'nullable|url',
            'position' => 'nullable|integer',
            'status' => 'nullable|boolean',
        ]);

        $data = [
            'title' => $request->title,
            'link' => $request->link,
            'position' => $request->position ?? 0,
            'status' => (bool) ($request->status ?? true),
        ];

        if ($request->hasFile('image')) {
            $data['image'] = self::imageUploadTrait($request->file('image'), $slider->image);
        }
        if ($request->hasFile('desktop_image')) {
            $data['desktop_image'] = self::imageUploadTrait($request->file('desktop_image'), $slider->desktop_image);
        }
        if ($request->hasFile('mobile_image')) {
            $data['mobile_image'] = self::imageUploadTrait($request->file('mobile_image'), $slider->mobile_image);
        }

        $slider->update($data);
        notify()->success(__('Slider updated'));
        return back();
    }

    public function destroy(Slider $slider): RedirectResponse
    {
        if (file_exists('assets/' . $slider->image)) {
            @unlink('assets/' . $slider->image);
        }
        $slider->delete();
        notify()->success(__('Slider deleted'));
        return back();
    }
}


