<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Transaction;
use App\Enums\TxnType;
use App\Enums\TxnStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class WalletController extends Controller
{
    /**
     * Display wallet management page
     */
    public function index()
    {
        $query = request('query');

        $builder = User::select('id', 'username', 'first_name', 'last_name', 'email', 'balance', 'profit_balance')
            ->orderBy('created_at', 'desc');

        if ($query) {
            $builder->where(function ($q) use ($query) {
                $q->where('username', 'like', "%{$query}%")
                    ->orWhere('first_name', 'like', "%{$query}%")
                    ->orWhere('last_name', 'like', "%{$query}%")
                    ->orWhere('email', 'like', "%{$query}%");
            });
        }

        $users = $builder->paginate(20)->appends(['query' => $query]);

        return view('backend.wallet.index', compact('users'));
    }

    /**
     * Show user wallet details
     */
    public function show($id)
    {
        $user = User::findOrFail($id);
        
        $transactions = Transaction::where('user_id', $id)
            ->whereIn('type', [TxnType::Deposit, TxnType::Withdraw, TxnType::Investment, TxnType::Interest, TxnType::Referral, TxnType::Bonus, TxnType::Salary])
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get();

        return view('backend.wallet.show', compact('user', 'transactions'));
    }

    /**
     * Update user balance
     */
    public function updateBalance(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'wallet_type' => 'required|in:main,profit',
            'action' => 'required|in:increase,decrease',
            'amount' => 'required|numeric|min:0.01',
            'reason' => 'required|string|max:255'
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        try {
            DB::beginTransaction();

            $user = User::findOrFail($request->user_id);
            $amount = $request->amount;
            $walletType = $request->wallet_type;
            $action = $request->action;
            $reason = $request->reason;

            // Determine the amount (positive for increase, negative for decrease)
            $transactionAmount = $action === 'increase' ? $amount : -$amount;

            // Update the appropriate wallet
            if ($walletType === 'main') {
                $user->increment('balance', $transactionAmount);
                $transactionType = $action === 'increase' ? TxnType::Deposit : TxnType::Withdraw;
            } else {
                $user->increment('profit_balance', $transactionAmount);
                $transactionType = $action === 'increase' ? TxnType::Interest : TxnType::Withdraw;
            }

            // Create transaction record
            Transaction::create([
                'user_id' => $user->id,
                'type' => $transactionType,
                'amount' => $amount,
                'status' => TxnStatus::Success,
                'description' => $reason,
                'tnx' => strtoupper(uniqid()),
                'created_at' => now(),
                'updated_at' => now()
            ]);

            DB::commit();

            $actionText = $action === 'increase' ? 'increased' : 'decreased';
            $walletText = $walletType === 'main' ? 'main wallet' : 'profit wallet';
            
            notify()->success("User {$user->username}'s {$walletText} {$actionText} by $" . number_format($amount, 2), 'Success');
            
        } catch (\Exception $e) {
            DB::rollback();
            notify()->error('Failed to update balance: ' . $e->getMessage(), 'Error');
        }

        return redirect()->back();
    }

    /**
     * Get user transactions via AJAX
     */
    public function getUserTransactions(Request $request)
    {
        $userId = $request->get('user_id');
        
        $transactions = Transaction::where('user_id', $userId)
            ->whereIn('type', [TxnType::Deposit, TxnType::Withdraw, TxnType::Investment, TxnType::Interest, TxnType::Referral, TxnType::Bonus, TxnType::Salary])
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get();

        return response()->json([
            'success' => true,
            'transactions' => $transactions
        ]);
    }

    /**
     * Search users
     */
    public function search(Request $request)
    {
        $query = $request->get('query');
        
        $users = User::where('username', 'like', "%{$query}%")
            ->orWhere('first_name', 'like', "%{$query}%")
            ->orWhere('last_name', 'like', "%{$query}%")
            ->orWhere('email', 'like', "%{$query}%")
            ->select('id', 'username', 'first_name', 'last_name', 'email', 'balance', 'profit_balance')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return view('backend.wallet.index', compact('users'));
    }
}





