<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Ranking;
use App\Models\User;
use App\Models\Invest;
use App\Models\Schema;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Carbon\Carbon;

class AchievementController extends Controller
{
    public function index()
    {
        try {
        $user = auth()->user();
        
            // ✅ DEBUG: Log user info
            \Log::info('Achievement Controller - User ID: ' . ($user ? $user->id : 'null'));
            
            // ✅ Initialize default values to prevent blank page
            $allRankings = collect([]);
            $currentRank = null;
            $currentRankName = 'NO RANK';
            $nextRank = null;
            $nextRankName = 'NO RANK';
            $totalReferrals = 0;
            $totalEarnings = 0;
            $totalDeposit = 0;
            $totalInvest = 0;
            $teamInvestment = 0;
            $totalReferralDeposits = 0;
            $totalReferralInvestments = 0;
            $progressPercentage = 0;
            $currency = 'USD';
            
            // ✅ SAFE: Ensure user has a starting rank with error handling
            try {
        if ($user->ranking_id === null) {
                    $defaultRank = \App\Models\Ranking::where('ranking_name', 'NO RANK')->where('ranking', 0)->first();
            if ($defaultRank) {
                $user->update(['ranking_id' => $defaultRank->id]);
                $user->refresh();
            }
                }
                \Log::info('Achievement Controller - User ranking_id: ' . $user->ranking_id);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Default rank assignment failed: ' . $e->getMessage());
        }
        
            // ✅ SAFE: Get all rankings with error handling
            try {
        $allRankings = \App\Models\Ranking::where('status', 1)->orderBy('id')->get();
                \Log::info('Achievement Controller - Found ' . $allRankings->count() . ' rankings');
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Rankings fetch failed: ' . $e->getMessage());
                $allRankings = collect([]);
            }
        
            // ✅ SAFE: Get current user's rank information
            try {
        $currentRank = $user->rank;
        $currentRankName = $currentRank ? $currentRank->ranking_name : 'NO RANK';
                \Log::info('Achievement Controller - Current rank: ' . $currentRankName);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Current rank fetch failed: ' . $e->getMessage());
            }
        
            // ✅ SAFE: Get next rank with error handling
            try {
                // Some databases store 'ranking' as text like "Level 1"; avoid numeric comparison issues
                $currentRankModel = optional($user)->rank;
                if ($currentRankModel) {
                    $nextRank = \App\Models\Ranking::where('status', 1)
                        ->where('id', '>', $currentRankModel->id)
                        ->orderBy('id', 'asc')
                        ->first();
                } else {
                    // User has no rank: pick the first active rank excluding NO RANK
                    $nextRank = \App\Models\Ranking::where('status', 1)
                        ->where('ranking_name', '!=', 'NO RANK')
                        ->orderBy('id', 'asc')
                        ->first();
                }
                $nextRankName = $nextRank ? $nextRank->ranking_name : 'NO RANK';
                \Log::info('Achievement Controller - Next rank: ' . $nextRankName);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Next rank fetch failed: ' . $e->getMessage());
            }
            
            // ✅ SAFE: Calculate user statistics with error handling
            try {
                $totalReferrals = \DB::table('users')->where('ref_id', $user->id)->count() ?? 0;
                \Log::info('Achievement Controller - Total referrals: ' . $totalReferrals);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Total referrals calculation failed: ' . $e->getMessage());
                $totalReferrals = 0;
            }
            
            try {
                $totalEarnings = $user->totalProfit() ?? 0;
                \Log::info('Achievement Controller - Total earnings: ' . $totalEarnings);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Total earnings calculation failed: ' . $e->getMessage());
                $totalEarnings = 0;
            }
            
            try {
                $totalDeposit = $user->totalDeposit() ?? 0;
                \Log::info('Achievement Controller - Total deposit: ' . $totalDeposit);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Total deposit calculation failed: ' . $e->getMessage());
                $totalDeposit = 0;
            }
            
            try {
                $totalInvest = $user->totalInvestment() ?? 0;
                \Log::info('Achievement Controller - Total invest: ' . $totalInvest);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Total investment calculation failed: ' . $e->getMessage());
                $totalInvest = 0;
            }
            
            // ✅ SAFE: Calculate team statistics with error handling
            try {
        $teamInvestment = $this->calculateTeamInvestment($user);
                \Log::info('Achievement Controller - Team investment: ' . $teamInvestment);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Team investment calculation failed: ' . $e->getMessage());
                $teamInvestment = 0;
            }
            
            try {
        $totalReferralDeposits = $this->calculateReferralDeposits($user);
                \Log::info('Achievement Controller - Referral deposits: ' . $totalReferralDeposits);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Referral deposits calculation failed: ' . $e->getMessage());
                $totalReferralDeposits = 0;
            }
            
            try {
        $totalReferralInvestments = $this->calculateReferralInvestments($user);
                \Log::info('Achievement Controller - Referral investments: ' . $totalReferralInvestments);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Referral investments calculation failed: ' . $e->getMessage());
                $totalReferralInvestments = 0;
            }
        
            // ✅ SAFE: Calculate progress percentage with error handling
            try {
        $progressPercentage = $this->calculateProgressPercentage($user, $currentRank, $totalReferrals, $teamInvestment, $totalReferralDeposits, $totalReferralInvestments);
                \Log::info('Achievement Controller - Progress percentage: ' . $progressPercentage);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Progress percentage calculation failed: ' . $e->getMessage());
                $progressPercentage = 0;
            }
        
            // ✅ SAFE: Calculate progress ring values
        $progressCircumference = 2 * M_PI * 35; // radius = 35
        $progressOffset = $progressCircumference - ($progressPercentage / 100) * $progressCircumference;
        
            // ✅ SAFE: Get currency with fallback
            try {
                $currency = setting('site_currency', 'global') ?? 'USD';
                \Log::info('Achievement Controller - Currency: ' . $currency);
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Currency setting failed: ' . $e->getMessage());
                $currency = 'USD';
            }
            
            // ✅ SAFE: Get referral users with stats for display
            try {
                $referralUsers = $this->getReferralUsersWithStats($user, 0);
                \Log::info('Achievement Controller - Referral users count: ' . $referralUsers->count());
            } catch (\Exception $e) {
                \Log::error('Achievement Controller - Referral users fetch failed: ' . $e->getMessage());
                $referralUsers = collect();
            }
            
            \Log::info('Achievement Controller - Successfully loaded all data');
            
            return view('frontend.shahdeveloper.achievement.index', compact(
                'user',
                'allRankings',
                'currentRank',
                'currentRankName',
                'nextRank',
                'nextRankName',
                'totalReferrals',
                'totalEarnings',
                'totalDeposit',
                'totalInvest',
                'teamInvestment',
                'totalReferralDeposits',
                'totalReferralInvestments',
                'progressPercentage',
                'progressCircumference',
                'progressOffset',
                'currency',
                'referralUsers'
            ))->with('rankings', $allRankings)->with('userRanking', $currentRank);
            
        } catch (\Exception $e) {
            // ✅ GLOBAL ERROR HANDLER: If anything fails, show default values
            \Log::error('Achievement Controller - Critical error: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            
            // Return safe default values to prevent blank page
            $user = auth()->user();
            $allRankings = collect([]);
            $currentRank = null;
            $currentRankName = 'NO RANK';
            $nextRank = null;
            $nextRankName = 'NO RANK';
            $totalReferrals = 0;
            $totalEarnings = 0;
            $totalDeposit = 0;
            $totalInvest = 0;
            $teamInvestment = 0;
            $totalReferralDeposits = 0;
            $totalReferralInvestments = 0;
            $progressPercentage = 0;
            $progressCircumference = 2 * M_PI * 35;
            $progressOffset = $progressCircumference;
            $currency = 'USD';
        
        return view('frontend.shahdeveloper.achievement.index', compact(
            'user',
            'allRankings',
            'currentRank',
            'currentRankName',
            'nextRank',
            'nextRankName',
            'totalReferrals',
            'totalEarnings',
            'totalDeposit',
            'totalInvest',
            'teamInvestment',
            'totalReferralDeposits',
            'totalReferralInvestments',
            'progressPercentage',
            'progressCircumference',
            'progressOffset',
            'currency'
        ))->with('rankings', $allRankings)->with('userRanking', $currentRank);
        }
    }
    
    /**
     * Calculate total earnings from referral users
     */
    private function calculateTotalEarnings($referralUsers)
    {
        $totalEarnings = 0;
        
        foreach ($referralUsers as $referralUser) {
            // ✅ FIXED: Calculate earnings from actual investments and profits
            $userInvestment = $referralUser->total_invest ?? 0;
            $userProfit = $referralUser->total_profit ?? 0;
            $userDeposit = $referralUser->total_deposit ?? 0;
            
            // Add user's investment amount and profits
            $totalEarnings += $userInvestment + $userProfit + $userDeposit;
        }
        
        return $totalEarnings;
    }

    /**
     * Get referral users with their statistics
     */
    private function getReferralUsersWithStats($user, $minimumInvest)
    {
        // ✅ FIXED: Get referral users with comprehensive investment data using direct SQL
        $sql = "SELECT 
            users.id, 
            users.username, 
            users.email, 
            users.balance,
            users.created_at,
            (SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id AND status = 'ongoing') as total_invest,
            (SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id) as total_invest_all,
            (SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = 'success' AND type IN ('deposit', 'manual_deposit')) as total_deposit,
            (SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = 'success' AND type = 'profit') as total_profit,
            (SELECT CASE WHEN (SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id AND status = 'ongoing') >= " . $minimumInvest . " THEN true ELSE false END) as requirementMet
        FROM users 
        WHERE users.ref_id = " . $user->id;
        
        $referralUsers = collect(\DB::select($sql));
            
        return $referralUsers;
    }
    
    public function checkAndUpdateRank()
    {
        $user = Auth::user();
        if (!$user) {
            return response()->json(['success' => false, 'message' => 'User not authenticated']);
        }
        
        $currentRank = $user->rank;
        $currentSequence = $currentRank ? $currentRank->ranking : 0;
        $nextRank = Ranking::where('status', 1)
            ->where('ranking', '>', $currentSequence)
            ->orderBy('ranking', 'asc')
            ->first();
            
        if (!$nextRank) {
            return response()->json(['success' => false, 'message' => 'No next rank available']);
        }
        
        // Get minimum investment requirement for next rank
        $minimumInvest = $nextRank->minimum_invest;
        
        // Get referral users with their stats
        $referralUsers = $this->getReferralUsersWithStats($user, $minimumInvest);
        
        // Check if all referrals have met their required sales
        $allRequirementsMet = true;
        foreach ($referralUsers as $referralUser) {
            if (!$referralUser->requirementMet) {
                $allRequirementsMet = false;
                break;
            }
        }
        
        // If all requirements are met, update the user's rank and add bonus (single-step promotion)
        if ($allRequirementsMet && count($referralUsers) > 0) {
            // Store previous rankings in the JSON array
            $rankings = json_decode($user->rankings, true) ?? [];
            if ($currentRank) {
                $rankings[] = $currentRank->id;
            }
            
            // Add bonus from current rank to user's balance
            $bonusAmount = $currentRank ? $currentRank->bonus : 0;
            if ($bonusAmount > 0) {
                $user->balance += $bonusAmount;
            }
            
            // Update user's ranking (single step)
            $user->ranking_id = $nextRank->id;
            $user->rankings = json_encode($rankings);
            $user->save();
            
            return response()->json([
                'success' => true, 
                'message' => 'Congratulations! You have been promoted to ' . $nextRank->ranking_name . 
                            ($bonusAmount > 0 ? ' and received a bonus of ' . $bonusAmount . ' ' . setting('site_currency', 'global') : ''),
                'newRank' => $nextRank->ranking_name,
                'bonusAdded' => $bonusAmount
            ]);
        }
        
        return response()->json([
            'success' => false, 
            'message' => 'Requirements not met yet for promotion'
        ]);
    }
    
    public function processRankUp(Request $request)
    {
        $user = Auth::user();
        if (!$user) {
            return redirect()->route('login');
        }
        
        $currentRankId = $request->input('current_rank_id');
        $nextRankId = $request->input('next_rank_id');
        
        $currentRank = Ranking::find($currentRankId);
        $nextRank = Ranking::find($nextRankId);
        
        if (!$nextRank) {
            return redirect()->back()->with('error', 'Invalid rank information');
        }
        
        // Get minimum investment requirement for current rank
        $minimumInvest = $currentRank ? $currentRank->minimum_invest : 900;
        
        // Get referral users with their stats
        $referralUsers = $this->getReferralUsersWithStats($user, $minimumInvest);
        $totalReferralCount = count($referralUsers);
        
        // Calculate the required amount per referral user
        $perUserRequiredAmount = $totalReferralCount > 0 ? $minimumInvest / $totalReferralCount : $minimumInvest;
        
        // Get default investment schema for rank-up contributions
        $defaultSchema = Schema::where('status', 1)->first();
        
        // If no schema found, we'll just deduct without creating investments
        if (!$defaultSchema) {
            // Just deduct the contribution amount from each referral user's balance
            foreach ($referralUsers as $referralUser) {
                $referralUserObj = User::find($referralUser->id);
                if ($referralUserObj && $referralUserObj->balance > 0) {
                    $deductionAmount = min($referralUserObj->balance, $perUserRequiredAmount);
                    $referralUserObj->balance -= $deductionAmount;
                    $referralUserObj->save();
                }
            }
            
            // Add bonus and update rank
            $bonusAmount = $currentRank ? $currentRank->bonus : 0;
            if ($bonusAmount > 0) {
                $user->balance += $bonusAmount;
            }
            
            $user->ranking_id = $nextRank->id;
            $rankings = json_decode($user->rankings, true) ?? [];
            if ($currentRank && !in_array($currentRank->id, $rankings)) {
                $rankings[] = $currentRank->id;
                $user->rankings = json_encode($rankings);
            }
            
            $user->save();
            
            return redirect()->route('user.referral.users')->with('success', 
                'Congratulations! You have been promoted to ' . $nextRank->ranking_name . 
                ($bonusAmount > 0 ? ' and received a bonus of ' . number_format($bonusAmount, 2) . ' ' . setting('site_currency', 'global') : '')
            );
        }
        
        // Begin transaction to ensure data integrity
        DB::beginTransaction();
        
        try {
            // Deduct the contribution amount from each referral user's current sale
            foreach ($referralUsers as $referralUser) {
                // Get the actual user object from the database
                $referralUserObj = User::find($referralUser->id);
                if ($referralUserObj) {
                    // Calculate how much to deduct from this user's balance
                    $referralUserBalance = $referralUserObj->balance ?? 0;
                    
                    // Only deduct if there's a balance to deduct from
                    if ($referralUserBalance > 0) {
                        // Calculate how much to deduct from this user's balance
                        // We'll deduct up to the required amount, but not more than their balance
                        $deductionAmount = min($referralUserBalance, $perUserRequiredAmount);
                        
                        if ($deductionAmount > 0) {
                            // Update the user's balance
                            $referralUserObj->balance -= $deductionAmount;
                            $referralUserObj->save();
                            
                            // Create an investment record for this deduction
                            $transactionId = rand(10000, 99999); // Simple numeric transaction ID
                            
                            // Calculate next profit time based on schema settings
                            $nextProfitTime = Carbon::now()->addHours($defaultSchema->period_hours);
                            
                            // Create the investment record using direct SQL to avoid model validation issues
                            DB::table('invests')->insert([
                                'user_id' => $referralUserObj->id,
                                'schema_id' => $defaultSchema->id,
                                'transaction_id' => $transactionId,
                                'invest_amount' => $deductionAmount,
                                'already_return_profit' => 0,
                                'total_profit_amount' => 0,
                                'last_profit_time' => null,
                                'next_profit_time' => $nextProfitTime,
                                'capital_back' => $defaultSchema->capital_back ?? 1,
                                'interest' => $defaultSchema->interest ?? 0.5,
                                'interest_type' => $defaultSchema->interest_type ?? 'percentage',
                                'return_type' => $defaultSchema->return_type ?? 'period',
                                'number_of_period' => $defaultSchema->number_of_period ?? 30,
                                'period_hours' => $defaultSchema->period_hours ?? 24,
                                'wallet' => 'main',
                                'status' => 'ongoing',
                                'created_at' => Carbon::now(),
                                'updated_at' => Carbon::now()
                            ]);
                        }
                    }
                }
            }
            
            // Add bonus from current rank to user's balance
            $bonusAmount = $currentRank ? $currentRank->bonus : 0;
            if ($bonusAmount > 0) {
                $user->balance += $bonusAmount;
            }
            
            // Update rank information
            $user->ranking_id = $nextRank->id;
            
            // Add current rank to rankings json array
            $rankings = json_decode($user->rankings, true) ?? [];
            if ($currentRank && !in_array($currentRank->id, $rankings)) {
                $rankings[] = $currentRank->id;
                $user->rankings = json_encode($rankings);
            }
            
            $user->save();
            
            // Commit the transaction
            DB::commit();
            
            return redirect()->route('user.referral.users')->with('success', 
                'Congratulations! You have been promoted to ' . $nextRank->ranking_name . 
                ($bonusAmount > 0 ? ' and received a bonus of ' . number_format($bonusAmount, 2) . ' ' . setting('site_currency', 'global') : '')
            );
            
        } catch (\Exception $e) {
            // Something went wrong, rollback the transaction
            DB::rollBack();
            
            // Log the error for debugging
            \Log::error('Rank-up error: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());
            
            return redirect()->back()->with('error', 'An error occurred during rank-up process. Please contact support.');
        }
    }
    
    /**
     * Calculate team investment for achievement progress
     */
    private function calculateTeamInvestment($user)
    {
        try {
        return \DB::table('invests')
            ->join('users', 'invests.user_id', '=', 'users.id')
            ->where('users.ref_id', $user->id)
            ->where('invests.status', 'ongoing')
                ->sum('invests.invest_amount') ?? 0;
        } catch (\Exception $e) {
            \Log::error('Calculate team investment failed: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Calculate referral deposits
     */
    private function calculateReferralDeposits($user)
    {
        try {
        return \DB::table('transactions')
            ->join('users', 'transactions.user_id', '=', 'users.id')
            ->where('users.ref_id', $user->id)
            ->where('transactions.status', 'success')
            ->where('transactions.type', 'deposit')
                ->sum('transactions.amount') ?? 0;
        } catch (\Exception $e) {
            \Log::error('Calculate referral deposits failed: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Calculate referral investments
     */
    private function calculateReferralInvestments($user)
    {
        try {
        return \DB::table('invests')
            ->join('users', 'invests.user_id', '=', 'users.id')
            ->where('users.ref_id', $user->id)
                ->sum('invests.invest_amount') ?? 0;
        } catch (\Exception $e) {
            \Log::error('Calculate referral investments failed: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Calculate progress percentage for next rank based on admin panel ranking requirements
     */
    private function calculateProgressPercentage($user, $nextRank, $totalReferrals, $teamInvestment, $totalReferralDeposits, $totalReferralInvestments)
    {
        try {
        if (!$nextRank) {
            return 100; // Max rank achieved
        }
        
        $progress = 0;
        $totalRequirements = 0;
        
        // Check referral requirement
        if ($nextRank->minimum_referral > 0) {
            $totalRequirements++;
            if ($totalReferrals >= $nextRank->minimum_referral) {
                $progress++;
            }
        }
        
            // Check earnings requirement - REMOVED as field doesn't exist in database
            // if ($nextRank->minimum_earnings > 0) {
            //     $totalRequirements++;
            //     if ($user->totalProfit() >= $nextRank->minimum_earnings) {
            //         $progress++;
            //     }
            // }
        
        // Check deposit requirement
        if ($nextRank->minimum_deposit > 0) {
            $totalRequirements++;
            if ($user->totalDeposit() >= $nextRank->minimum_deposit) {
                $progress++;
            }
        }
        
        // Check investment requirement
        if ($nextRank->minimum_invest > 0) {
            $totalRequirements++;
            if ($user->totalInvestment() >= $nextRank->minimum_invest) {
                $progress++;
            }
        }
        
        // Check referral deposit requirement
        if ($nextRank->minimum_referral_deposit > 0) {
            $totalRequirements++;
            if ($totalReferralDeposits >= $nextRank->minimum_referral_deposit) {
                $progress++;
            }
        }
        
        // Check referral investment requirement
        if ($nextRank->minimum_referral_invest > 0) {
            $totalRequirements++;
            if ($totalReferralInvestments >= $nextRank->minimum_referral_invest) {
                $progress++;
            }
        }
        
        return $totalRequirements > 0 ? ($progress / $totalRequirements) * 100 : 0;
            
        } catch (\Exception $e) {
            \Log::error('Calculate progress percentage failed: ' . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Collect achievement reward
     */
    public function collectReward(Request $request)
    {
        $user = Auth::user();
        $rankId = $request->input('rank_id');
        
        if (!$user) {
            return response()->json(['success' => false, 'message' => 'User not authenticated']);
        }
        
        // Get ranking from admin panel
        $ranking = \App\Models\Ranking::find($rankId);
        if (!$ranking) {
            return response()->json(['success' => false, 'message' => 'Invalid ranking']);
        }
        
        // Check if user has already collected this reward
        $rankings = json_decode($user->rankings, true) ?? [];
        if (in_array($rankId, $rankings)) {
            return response()->json(['success' => false, 'message' => 'Reward already collected']);
        }
        
        // Check if user meets requirements
        $totalReferrals = \DB::table('users')->where('ref_id', $user->id)->count();
        $totalEarnings = $user->totalProfit();
        $totalDeposit = $user->totalDeposit();
        $totalInvest = $user->totalInvestment();
        $totalReferralDeposits = $this->calculateReferralDeposits($user);
        $totalReferralInvestments = $this->calculateReferralInvestments($user);
        
        $meetsRequirements = (
            ($ranking->minimum_referral == 0 || $totalReferrals >= $ranking->minimum_referral) &&
            ($ranking->minimum_earnings == 0 || $totalEarnings >= $ranking->minimum_earnings) &&
            ($ranking->minimum_deposit == 0 || $totalDeposit >= $ranking->minimum_deposit) &&
            ($ranking->minimum_invest == 0 || $totalInvest >= $ranking->minimum_invest) &&
            ($ranking->minimum_referral_deposit == 0 || $totalReferralDeposits >= $ranking->minimum_referral_deposit) &&
            ($ranking->minimum_referral_invest == 0 || $totalReferralInvestments >= $ranking->minimum_referral_invest)
        );
        
        if (!$meetsRequirements) {
            return response()->json(['success' => false, 'message' => 'Requirements not met for this rank']);
        }
        
        $rewardAmount = $ranking->bonus ?? 0;
        
        if ($rewardAmount > 0) {
            // Add reward to user's balance
            $user->balance += $rewardAmount;
            
            // Add rank to completed rankings
            $rankings[] = $rankId;
            $user->rankings = json_encode($rankings);
            $user->save();
            
            return response()->json([
                'success' => true,
                'message' => 'Reward collected successfully!',
                'amount' => $rewardAmount
            ]);
        }
        
        return response()->json(['success' => false, 'message' => 'No reward available for this rank']);
    }
    
    /**
     * Collect monthly salary (for users with salary-eligible ranks)
     */
    public function collectSalary(Request $request)
    {
        $user = Auth::user();
        
        if (!$user) {
            return response()->json(['success' => false, 'message' => 'User not authenticated']);
        }
        
        // Get all rankings with salary from admin panel
        $salaryEligibleRankings = \App\Models\Ranking::where('salary', '>', 0)->get();
        $completedRankings = json_decode($user->rankings, true) ?? [];
        
        // Find highest salary-eligible rank user has completed
        $eligibleRank = null;
        foreach($salaryEligibleRankings as $ranking) {
            if(in_array($ranking->id, $completedRankings)) {
                $eligibleRank = $ranking;
            }
        }
        
        if (!$eligibleRank) {
            return response()->json(['success' => false, 'message' => 'You must complete a rank with salary benefits to collect salary']);
        }
        
        // Check if salary was already collected this month
        $lastSalaryDate = $user->last_salary_date;
        $currentMonth = Carbon::now()->format('Y-m');
        
        if ($lastSalaryDate && Carbon::parse($lastSalaryDate)->format('Y-m') === $currentMonth) {
            return response()->json(['success' => false, 'message' => 'Salary already collected this month']);
        }
        
        // Add salary to user's balance
        $salaryAmount = $eligibleRank->salary;
        $user->balance += $salaryAmount;
        $user->last_salary_date = Carbon::now();
        $user->save();
        
        return response()->json([
            'success' => true,
            'message' => 'Monthly salary collected successfully!',
            'amount' => $salaryAmount
        ]);
    }
}