<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Models\DepositMethod;
use App\Models\Transaction;
use App\Traits\ImageUpload;
use App\Traits\NotifyTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Txn;
use Validator;

class DepositController extends GatewayController
{
    use ImageUpload, NotifyTrait;

    public function deposit()
    {
        if (!setting('user_deposit', 'permission')) {
            abort('403', 'Deposit Disable Now');
        }
        
        if (!\Auth::check()) {
            return redirect()->route('login');
        }
        
        if (!\Auth::user()->deposit_status) {
            abort('403', 'Your deposit is disabled');
        }

        // Load gateways from admin-managed DepositMethod
        $gateways = DepositMethod::where('status', 1)->get();
        return view('frontend.shahdeveloper.deposit.index', compact('gateways'));
    }

    public function depositNow(Request $request)
    {
        \Log::info('Deposit Now Request:', $request->all());

        if (!setting('user_deposit', 'permission')) {
            abort('403', 'Deposit Disable Now');
        }
        
        if (!\Auth::check()) {
            return redirect()->route('login');
        }
        
        if (!\Auth::user()->deposit_status) {
            abort('403', 'Your deposit is disabled');
        }

        $validator = Validator::make($request->all(), [
            'gateway_code' => 'required|string',
            'amount' => ['required', 'regex:/^[0-9]+(\.[0-9][0-9]?)?$/'],
        ]);

        if ($validator->fails()) {
            \Log::error('Deposit Validation Failed:', $validator->errors()->toArray());
            notify()->error($validator->errors()->first(), 'Error');

            return redirect()->back();
        }

        $input = $request->all();
        \Log::info('Deposit Input Data:', $input);

        $gatewayInfo = DepositMethod::code($input['gateway_code'])->first();
        \Log::info('Gateway Info:', $gatewayInfo ? (array)$gatewayInfo : null);
        $amount = $input['amount'];
        \Log::info('Amount from input:', ['amount' => $amount]);
        
        // Debug: Log the exact amount being processed
        \Log::info('Processing amount:', ['original_amount' => $amount, 'type' => gettype($amount)]);

        if (!$gatewayInfo) {
            notify()->error('Invalid payment method selected', 'Error');
            return redirect()->back();
        }

        $min = (float)$gatewayInfo->minimum_deposit;
        $max = (float)$gatewayInfo->maximum_deposit;
        if ($amount < $min || $amount > $max) {
            $currencySymbol = setting('currency_symbol', 'global');
            $message = 'Please Deposit the Amount within the range ' . $currencySymbol . $min . ' to ' . $currencySymbol . $max;
            notify()->error($message, 'Error');

            return redirect()->back();
        }

        // Compute from DepositMethod
        $charge = $gatewayInfo->charge_type == 'percentage' ? (($gatewayInfo->charge / 100) * $amount) : $gatewayInfo->charge;
        $rate = $gatewayInfo->rate ?? 1;
        $finalAmount = (float)$amount + (float)$charge;
        $payAmount = $finalAmount * $rate;

        // Store deposit data in session (NO transaction created yet)
        $depositData = [
            'amount' => $amount,
            'charge' => $charge,
            'final_amount' => $finalAmount,
            'pay_amount' => $payAmount,
            'gateway_code' => $gatewayInfo->gateway_code,
            'gateway_name' => $gatewayInfo->name,
            'rate' => $rate,
            'currency' => $gatewayInfo->currency ?? 'USD',
            'user_id' => auth()->id(),
            'created_at' => now()->toDateTimeString(),
        ];

        // Store in session for proof submission
        session(['deposit_data' => $depositData]);
        
        \Log::info('Deposit data stored in session:', $depositData);

        // Redirect to proof submission page (NO transaction created yet)
        notify()->success('Please complete your payment and submit proof for approval.', 'Success');
        return redirect()->route('user.deposit.proof');

    }

    public function depositLog()
    {
        $deposits = Transaction::search(request('query'), function ($query) {
            $query->where('user_id', auth()->user()->id)
                ->when(request('date'), function ($query) {
                    $query->whereDay('created_at', '=', Carbon::parse(request('date'))->format('d'));
                })
                ->whereIn('type', [TxnType::Deposit, TxnType::ManualDeposit]);
        })->orderBy('created_at', 'desc')->paginate(10)->withQueryString();

        return view('frontend::deposit.log', compact('deposits'));
    }

    public function depositProof()
    {
        if (!setting('user_deposit', 'permission') || !\Auth::user()->deposit_status) {
            abort('403', 'Deposit Disable Now');
        }

        // Check if deposit data exists in session
        $depositData = session('deposit_data');
        if (!$depositData) {
            notify()->error('No deposit request found. Please start a new deposit.', 'Error');
            return redirect()->route('user.deposit');
        }

        // Load gateways from admin-managed DepositMethod for proof selection
        $gateways = DepositMethod::where('status', 1)->get();
        
        // Create transaction object from session data for display
        $transaction = (object) $depositData;
        $transaction->tnx = 'PENDING'; // Will be generated when proof is submitted
        $transaction->status = 'draft'; // Not yet submitted
        
        return view('frontend.shahdeveloper.user.deposit-proof', compact('gateways', 'transaction'));
    }

    public function getGatewayDetails(Request $request)
    {
        $gatewayId = $request->gateway_id;
        // Fetch from DepositMethod (admin panel)
        $gateway = DepositMethod::find($gatewayId);
        
        if (!$gateway) {
            return response()->json(['success' => false, 'message' => 'Gateway not found']);
        }
        
        $paymentDetails = json_decode($gateway->payment_details, true);
        
        if (!$paymentDetails) {
            return response()->json([
                'success' => false,
                'message' => 'No payment details configured for this gateway. Please contact administrator.'
            ]);
        }
        
        return response()->json([
            'success' => true,
            'wallet_address' => $paymentDetails['wallet_address'] ?? 'No wallet address available',
            'qr_code' => $paymentDetails['qr_code'] ?? 'https://api.qrserver.com/v1/create-qr-code/?size=150x150&data=Sample',
            'gateway_name' => $gateway->name
        ]);
    }

    public function depositSubmit(Request $request)
    {
        if (!setting('user_deposit', 'permission') || !\Auth::user()->deposit_status) {
            abort('403', 'Deposit Disable Now');
        }

        \Log::info('Deposit Submit Request:', $request->all());
        \Log::info('Files:', $request->files->all());

        $validator = Validator::make($request->all(), [
            'transaction_hash' => 'required|string',
            'proof_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120', // 5MB max
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            \Log::error('Validation failed:', $validator->errors()->toArray());
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        try {
            // Get deposit data from session
            $depositData = session('deposit_data');
            if (!$depositData) {
                \Log::error('No deposit data found in session');
                notify()->error('No deposit request found. Please start a new deposit.', 'Error');
                return redirect()->route('user.deposit');
            }

            // Upload proof file
            $proofFile = $this->imageUploadTrait($request->file('proof_file'));
            
            // Create manual field data for admin panel display
            $manualFieldData = [
                'Payment Proof' => $proofFile,
                'Transaction Hash' => $request->transaction_hash,
                'Notes' => $request->notes ?? 'No additional notes provided'
            ];
            
            // Generate unique transaction ID
            $txnId = 'TXN' . time() . rand(1000, 9999);
            
            // Create transaction NOW (after proof submission)
            $transaction = new \App\Models\Transaction();
            $transaction->user_id = $depositData['user_id'];
            $transaction->tnx = $txnId;
            $transaction->amount = $depositData['amount'];
            $transaction->charge = $depositData['charge'];
            $transaction->final_amount = $depositData['final_amount'];
            $transaction->method = $depositData['gateway_code'];
            $transaction->description = 'Deposit With ' . $depositData['gateway_name'];
            $transaction->type = TxnType::ManualDeposit;
            $transaction->status = TxnStatus::Pending->value; // Now pending after proof submission
            $transaction->pay_amount = $depositData['pay_amount'];
            $transaction->proof_file = $proofFile;
            $transaction->transaction_hash = $request->transaction_hash;
            $transaction->notes = $request->notes;
            $transaction->manual_field_data = json_encode($manualFieldData);
            $transaction->created_at = $depositData['created_at']; // Use original timestamp
            $transaction->save();

            \Log::info('Transaction created with proof:', ['txn_id' => $txnId, 'amount' => $transaction->amount, 'proof_file' => $proofFile]);

            // Send notification to admin
            $adminMessage = 'User ' . auth()->user()->username . ' has submitted deposit proof for transaction ' . $txnId . ' (Amount: $' . $transaction->amount . ')';
            
            \Log::info('Sending admin notification:', ['message' => $adminMessage]);
            
            // Create admin notification
            try {
                \App\Models\Notification::create([
                    'user_id' => 1, // Admin user ID
                    'title' => 'New Deposit Proof Submitted',
                    'message' => $adminMessage,
                    'type' => 'deposit_proof',
                    'status' => 'unread',
                    'created_at' => now(),
                ]);
                \Log::info('Admin notification created successfully');
            } catch (\Exception $e) {
                \Log::error('Failed to create admin notification: ' . $e->getMessage());
            }

            if (request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Deposit submitted for approval successfully.',
                    'redirect_url' => route('user.deposit.success')
                ]);
            }
            
            // Clear session data
            session()->forget(['deposit_data']);
            
            // Success feedback and redirect to dedicated success page
            notify()->success('Your deposit proof has been submitted successfully. We\'re reviewing it shortly.', 'Success');
            return redirect()->route('user.deposit.success')->with('success', 'Deposit proof submitted successfully and is pending approval.');

        } catch (\Exception $e) {
            \Log::error('Deposit Submit Exception', [
                'message' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);
            if (request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 500);
            }
            
            notify()->error($e->getMessage(), 'Error');
            return redirect()->back();
        }
    }
}
