<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\InvestStatus;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Models\DepositMethod;
use App\Models\Invest;
use App\Models\LevelReferral;
use App\Models\Schema;
use App\Traits\ImageUpload;
use App\Traits\NotifyTrait;
use Auth;
use Carbon\Carbon;
use DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Txn;

class InvestController extends GatewayController
{
    use ImageUpload, NotifyTrait;

    public function investNow(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'schema_id' => 'required',
            'invest_amount' => 'regex:/^[0-9]+(\.[0-9][0-9]?)?$/',
            'wallet' => 'in:main,profit,gateway',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        $input = $request->all();
        $user = Auth::user();
        $schema = Schema::with('schedule')->find($input['schema_id']);

        if (!$schema) {
            notify()->error('Invalid Plan Selected', 'Error');
            return redirect()->back();
        }

        $investAmount = $input['invest_amount'];

        // Insufficient Balance validation
        if ($input['wallet'] == 'main' && $user->balance < $investAmount) {
            notify()->error('Insufficient Balance in Your Main Wallet', 'Error');
            return redirect()->route('user.schema.preview', $schema->id);
        } elseif ($input['wallet'] == 'profit' && $user->profit_balance < $investAmount) {
            notify()->error('Insufficient Balance in Your Profit Wallet', 'Error');
            return redirect()->route('user.schema.preview', $schema->id);
        }

        // Invalid Amount check based on schema type
        if (($schema->type == 'range' && ($schema->min_amount > $investAmount || $schema->max_amount < $investAmount)) || ($schema->type == 'fixed' && $schema->fixed_amount != $investAmount)) {
            notify()->error('Invest Amount Out Of Range', 'Error');
            return redirect()->route('user.schema.preview', $schema->id);
        }

        $timezone = setting('site_timezone', 'global') ?? 'UTC';
        // use minutes-based schedule (convert to hours if needed)
        $scheduleMinutes = optional($schema->schedule)->time ?? 1440; // Default 24 hours in minutes
        $nextProfitTime = Carbon::now($timezone)->addMinutes($scheduleMinutes);

        $data = [
            'user_id' => $user->id,
            'schema_id' => $schema->id,
            'invest_amount' => $investAmount,
            'next_profit_time' => $nextProfitTime,
            'last_profit_time' => null, // Initialize as null
            'already_return_profit' => 0, // Initialize profit count
            'capital_back' => $schema->capital_back,
            'interest' => $schema->return_interest,
            'interest_type' => $schema->interest_type,
            'return_type' => $schema->return_type,
            'number_of_period' => $schema->number_of_period,
            'period_hours' => round($scheduleMinutes / 60, 1), // Convert minutes to hours for display
            'wallet' => $input['wallet'],
            'status' => InvestStatus::Ongoing,
        ];

        if ($input['wallet'] == 'main') {
            $user->decrement('balance', $investAmount);
        } elseif ($input['wallet'] == 'profit') {
            $user->decrement('profit_balance', $investAmount);
        } else {
            // Gateway payment flow
            $gatewayInfo = DepositMethod::code($input['gateway_code'])->first();

            if (!$gatewayInfo) {
                notify()->error('Invalid Payment Gateway', 'Error');
                return redirect()->back();
            }

            $charge = $gatewayInfo->charge_type == 'percentage' ? (($gatewayInfo->charge / 100) * $investAmount) : $gatewayInfo->charge;
            $finalAmount = (float)$investAmount + (float)$charge;
            $payAmount = $finalAmount * $gatewayInfo->rate;
            $payCurrency = $gatewayInfo->currency;

            $manualData = null;
            if (isset($input['manual_data'])) {
                $manualData = $input['manual_data'];
                foreach ($manualData as $key => $value) {
                    if (is_file($value)) {
                        $manualData[$key] = self::imageUploadTrait($value);
                    }
                }
            }

            $txnInfo = Txn::new(
                $investAmount,
                $charge,
                $finalAmount,
                $gatewayInfo->name,
                $schema->name . ' Invested',
                TxnType::Investment,
                TxnStatus::Pending,
                $payCurrency,
                $payAmount,
                $user->id,
                null,
                'user',
                $manualData ?? []
            );

            $data = array_merge($data, ['status' => InvestStatus::Pending, 'transaction_id' => $txnInfo->id]);
            $investment = Invest::create($data);
            
            // Create ROI schedule for this investment (will be activated when payment is confirmed)
            $this->createRoiSchedule($investment);

            return self::depositAutoGateway($input['gateway_code'], $txnInfo);
        }

        // Create successful txn for wallet invest
        $txnInfo = Txn::new(
            $investAmount,
            0,
            $investAmount,
            'system',
            $schema->name . ' Plan Invested',
            TxnType::Investment,
            TxnStatus::Success,
            null,
            null,
            $user->id
        );

        $data = array_merge($data, ['transaction_id' => $txnInfo->id]);
        $investment = Invest::create($data);
        
        // Create ROI schedule for this investment
        $this->createRoiSchedule($investment);

        // REMOVED: Investment referral commission disabled
        // Only deposit referral commission is allowed (one-time)

        $shortcodes = [
            '[[full_name]]' => $txnInfo->user->full_name,
            '[[txn]]' => $txnInfo->tnx,
            '[[plan_name]]' => $txnInfo->invest->schema->name,
            '[[invest_amount]]' => $txnInfo->amount . setting('site_currency', 'global'),
            '[[site_title]]' => setting('site_title', 'global'),
            '[[site_url]]' => route('home'),
        ];

        $this->mailNotify($txnInfo->user->email, 'user_investment', $shortcodes);
        $this->pushNotify('user_investment', $shortcodes, route('user.collect-roi'), $txnInfo->user->id);
        $this->smsNotify('user_investment', $shortcodes, $txnInfo->user->phone);

        notify()->success('Successfully Investment', 'success');
        return redirect()->route('user.collect-roi');
    }


    public function investLogs(Request $request)
    {
        $user = auth()->user();
        
        // Get all investments for the user
        $investments = Invest::where('user_id', $user->id)
            ->with(['schema'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Currency symbol
        $currencySymbol = setting('currency_symbol', 'global') ?? '$';

        // Calculate total invested amount (support legacy/new columns)
        $totalInvested = $investments->sum(function ($investment) {
            return (float) ($investment->invest_amount ?? $investment->amount ?? 0);
        });
        
        // Calculate total profit earned (support legacy/new columns)
        $totalProfit = $investments->sum(function ($investment) {
            return (float) ($investment->total_profit_amount ?? $investment->profit ?? 0);
        });
        
        // Normalize status for enum/string
        $normalizeStatus = function ($investment) {
            $status = $investment->status;
            if ($status instanceof \App\Enums\InvestStatus) {
                return $status->value;
            }
            return (string) $status;
        };
        
        // Calculate active and completed investments
        $activeInvestments = $investments->filter(function ($inv) use ($normalizeStatus) {
            return $normalizeStatus($inv) === 'ongoing';
        })->count();
        
        $completedInvestments = $investments->filter(function ($inv) use ($normalizeStatus) {
            return $normalizeStatus($inv) === 'completed';
        })->count();

        return view('frontend.shahdeveloper.user.invest-logs', compact(
            'investments', 
            'totalInvested', 
            'totalProfit', 
            'activeInvestments', 
            'completedInvestments',
            'currencySymbol'
        ));
    }

    public function investCancel($id)
    {
        $investment = Invest::with('schema')->find($id);

        if (!$investment) {
            notify()->error('Investment not found', 'Error');
            return redirect()->back();
        }

        // Check if user owns this investment
        if ($investment->user_id != auth()->id()) {
            notify()->error('Unauthorized access', 'Error');
            return redirect()->back();
        }

        // Check if investment is ongoing
        if ($investment->status != InvestStatus::Ongoing) {
            notify()->error('Only ongoing investments can be cancelled', 'Error');
            return redirect()->back();
        }

        // Check if schema allows cancellation
        if (!$investment->schema->schema_cancel) {
            notify()->error('This investment plan does not allow cancellation', 'Error');
            return redirect()->back();
        }

        // daily limit on cancel
        $todayTransaction = Invest::where('user_id', auth()->user()->id)
            ->where('status', InvestStatus::Canceled)
            ->whereDate('created_at', Carbon::today())
            ->count();

        $dayLimit = (float)setting('send_money_day_limit', 'fee');
        if ($todayTransaction >= $dayLimit) {
            notify()->error(__('Today Investment Cancel limit has been reached'), 'Error');
            return redirect()->back();
        }

        // Cancel the investment
        $investment->update(['status' => InvestStatus::Canceled]);

        $user = $investment->user;
        
        // Refund based on wallet type
        if ($investment->wallet == 'main') {
            $user->balance += $investment->invest_amount;
        } elseif ($investment->wallet == 'profit') {
            $user->profit_balance += $investment->invest_amount;
        }
        $user->save();

        // Create refund transaction
        Txn::new(
            $investment->invest_amount,
            0,
            $investment->invest_amount,
            'system',
            $investment->schema->name . ' Investment Cancelled - Refund to ' . ucfirst($investment->wallet) . ' Wallet',
            TxnType::Refund,
            TxnStatus::Success,
            null,
            null,
            $user->id
        );

        notify()->success('Investment cancelled successfully and amount refunded', 'Success');
        return redirect()->route('user.invest-logs');
    }

    // ROI collection removed - ROI will be sent by admin automatically
    
    /**
     * Create ROI schedule for investment
     */
    private function createRoiSchedule($investment)
    {
        $roiAmount = ($investment->invest_amount * $investment->schema->return_interest) / 100;
        $totalCycles = $investment->schema->number_of_period ?? 30;
        
        // Normalize schedule to minutes (DB may store hours like 24, or minutes like 1440)
        $rawTime = optional($investment->schema->schedule)->time ?? ($investment->period_hours ?: 24);
        $scheduleMinutes = $rawTime > 48 ? $rawTime : ($rawTime * 60);
        $anchorStart = \Carbon\Carbon::parse($investment->created_at);
        $anchorEnd = $anchorStart->copy()->addMinutes($scheduleMinutes);
        
        \App\Models\RoiSchedule::create([
            'invest_id' => $investment->id,
            'user_id' => $investment->user_id,
            'schema_id' => $investment->schema_id,
            'invest_amount' => $investment->invest_amount,
            'roi_percentage' => $investment->schema->return_interest,
            'roi_amount' => $roiAmount,
            'roi_start_time' => $anchorStart,
            'roi_end_time' => $anchorEnd,
            'status' => 'pending',
            'roi_cycle' => 1,
            'total_cycles' => $totalCycles
        ]);
    }
}