<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Invest;
use App\Models\Transaction;
use App\Models\Txn;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class InvestLogController extends Controller
{
    /**
     * Display invest logs page
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get all investments for the user
        $investments = Invest::where('user_id', $user->id)
            ->with(['schema', 'roiSchedules', 'activeRoiSchedule', 'readyRoiSchedule'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        // Add calculated end date to each investment
        $investments->each(function ($investment) {
            $investment->calculated_end_date = $this->calculateEndDate($investment);
        });
        
        // Calculate total invested amount
        $totalInvested = $investments->sum('invest_amount');
        
        // Calculate total profit earned
        $totalProfit = $investments->sum('total_profit_amount');
        
        // Calculate active investments
        $activeInvestments = $investments->where('status', 'active')->count();
        
        // Calculate completed investments
        $completedInvestments = $investments->where('status', 'completed')->count();
        
        return view('frontend.shahdeveloper.user.invest-logs', compact(
            'investments', 
            'totalInvested', 
            'totalProfit', 
            'activeInvestments', 
            'completedInvestments'
        ));
    }

    /**
     * Collect ROI for completed investment
     */
    public function collectRoi(Request $request)
    {
        $request->validate([
            'invest_id' => 'required|exists:invests,id'
        ]);
        
        $user = Auth::user();
        $invest = Invest::where('id', $request->invest_id)
            ->where('user_id', $user->id)
            ->first();
        
        if (!$invest) {
            return response()->json([
                'success' => false,
                'message' => 'Investment not found'
            ]);
        }
        
        // Check if investment is completed
        if ($invest->status !== 'completed') {
            return response()->json([
                'success' => false,
                'message' => 'Investment is not completed yet'
            ]);
        }
        
        // Check if ROI is already collected
        if ($invest->roi_collected) {
            return response()->json([
                'success' => false,
                'message' => 'ROI already collected'
            ]);
        }

        // Additional check for recent collection (within 1 minute)
        $now = Carbon::now();
        $lastProfitTime = $invest->last_profit_time ? Carbon::parse($invest->last_profit_time) : null;
        
        if ($lastProfitTime && $lastProfitTime->diffInMinutes($now) < 1) {
            return response()->json([
                'success' => false,
                'message' => 'ROI already collected recently. Please wait.'
            ]);
        }
        
        // Calculate total profit based on SCHEMA settings
        $totalProfit = $this->calculateRoiFromSchema($invest);
        
        if ($totalProfit <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'No profit to collect'
            ]);
        }
        
        // Add profit to user's profit wallet
        $user->increment('profit_balance', $totalProfit);
        
        // Mark ROI as collected and update last profit time
        $invest->update([
            'roi_collected' => true,
            'last_profit_time' => $now
        ]);
        
        // Create transaction record
        Txn::new(
            $totalProfit,
            0, // No charge
            $totalProfit,
            'ROI-' . strtoupper(\Str::random(10)),
            'ROI Collection from ' . $invest->schema->name,
            TxnType::Interest,
            TxnStatus::Success,
            setting('site_currency', 'global'),
            $totalProfit,
            $user->id,
            null,
            'System',
            [
                'invest_id' => $invest->id,
                'schema_name' => $invest->schema->name,
                'original_amount' => $invest->amount,
                'profit_amount' => $totalProfit
            ]
        );
        
        return response()->json([
            'success' => true,
            'message' => 'ROI collected successfully!',
            'profit' => $totalProfit
        ]);
    }

    /**
     * Calculate ROI progress percentage for an investment
     */
    private function calculateRoiProgress($investment)
    {
        if ($investment->status !== 'ongoing') {
            return 100; // Completed investments show 100%
        }

        $now = Carbon::now();
        $nextProfitTime = Carbon::parse($investment->next_profit_time);
        
        // If ROI is ready (next profit time has passed)
        if ($nextProfitTime <= $now) {
            return 100;
        }

        // Get the last profit time or investment creation time
        $lastProfitTime = $investment->last_profit_time ? 
            Carbon::parse($investment->last_profit_time) : 
            Carbon::parse($investment->created_at);

        // Calculate total time between last profit and next profit
        $totalTime = $lastProfitTime->diffInMinutes($nextProfitTime);
        $elapsedTime = $lastProfitTime->diffInMinutes($now);

        if ($totalTime <= 0) {
            return 100;
        }

        $progress = ($elapsedTime / $totalTime) * 100;
        return min(100, max(0, round($progress, 1)));
    }

    /**
     * Calculate end date based on pending periods
     */
    private function calculateEndDate($investment)
    {
        if ($investment->status === 'completed') {
            return $investment->end_date ? Carbon::parse($investment->end_date) : null;
        }

        if ($investment->return_type === 'lifetime') {
            return null; // No end date for lifetime investments
        }

        // Calculate pending periods
        $completedPeriods = $investment->already_return_profit ?? 0;
        $totalPeriods = $investment->schema->number_of_period ?? 0;
        $pendingPeriods = $totalPeriods - $completedPeriods;

        if ($pendingPeriods <= 0) {
            return Carbon::now(); // Already completed
        }

        // Get schedule time in minutes
        $scheduleMinutes = $investment->schema->schedule->time ?? 1440; // Default 24 hours

        // Calculate end date based on pending periods
        $lastProfitTime = $investment->last_profit_time ? 
            Carbon::parse($investment->last_profit_time) : 
            Carbon::parse($investment->created_at);

        // Add remaining periods * schedule time
        $endDate = $lastProfitTime->copy()->addMinutes($pendingPeriods * $scheduleMinutes);

        return $endDate;
    }

    /**
     * Calculate ROI based on current schema settings
     */
    private function calculateRoiFromSchema($investment)
    {
        try {
            $schema = $investment->schema;
            
            if (!$schema) {
                // Fallback to original investment calculation
                return $investment->profit ?? 0;
            }
            
            $roiAmount = 0;
            
            // Use SCHEMA settings for ROI calculation
            if ($schema->interest_type === 'percentage') {
                // Percentage-based ROI from schema
                $roiAmount = ($schema->return_interest * $investment->invest_amount) / 100;
            } else {
                // Fixed amount ROI from schema
                $roiAmount = $schema->return_interest;
            }
            
            // Use actual schedule time from database
            $scheduleMinutes = $schema->schedule->time ?? 1440; // Default to 24 hours (in minutes)
            
            // For different frequency plans, adjust ROI accordingly
            if ($scheduleMinutes < 1440) { // Less than 24 hours
                $roiAmount = ($roiAmount * $scheduleMinutes) / 1440; // Convert to actual frequency rate
            }
            
            return round($roiAmount, 2);
            
        } catch (\Exception $e) {
            \Log::error('ROI calculation error: ' . $e->getMessage());
            // Fallback to original investment calculation
            return $investment->profit ?? 0;
        }
    }
}