<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Invest;
use App\Models\Transaction;
use App\Enums\TxnType;
use App\Enums\TxnStatus;
use App\Enums\InvestStatus;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class InvestmentCancelController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    public function debugCancel(Request $request)
    {
        try {
            return response()->json([
                'success' => true,
                'message' => 'Debug route working',
                'user_id' => auth()->id(),
                'request_data' => $request->all()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Debug route failed: ' . $e->getMessage(),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function cancelInvestment(Request $request)
    {
        try {
            \Log::info('Investment cancellation request started', [
                'request_data' => $request->all(),
                'user_id' => auth()->id()
            ]);

            // Auth check
            if (!auth()->check()) {
                \Log::error('User not authenticated');
                return response()->json(['success' => false, 'message' => 'User not authenticated'], 401);
            }

            $request->validate([
                'investment_id' => 'required|exists:invests,id',
                'reason' => 'nullable|string|max:1000'
            ]);

            $user = auth()->user();
            $investment = Invest::where('id', $request->investment_id)
                ->where('user_id', $user->id)
                ->where('status', 'ongoing')
                ->with('schema')
                ->first();

            \Log::info('Investment found', [
                'investment' => $investment ? $investment->toArray() : 'Not found',
                'schema' => $investment && $investment->schema ? $investment->schema->toArray() : 'No schema'
            ]);

            if (!$investment) {
                return response()->json(['success' => false, 'message' => 'Investment not found or not eligible for cancellation'], 404);
            }

            // Cancellable?
            if (!isset($investment->schema->schema_cancel) || $investment->schema->schema_cancel != 1) {
                return response()->json(['success' => false, 'message' => 'This investment plan does not allow cancellation'], 400);
            }

            if ($investment->is_cancelled) {
                return response()->json(['success' => false, 'message' => 'Investment is already cancelled'], 400);
            }

            \Log::info('Starting transaction for investment cancellation', [
                'investment_id' => $investment->id,
                'user_id' => $user->id
            ]);

            DB::beginTransaction();

            // NEW LOGIC: Calculate deduction based on time
            $investmentDate = Carbon::parse($investment->created_at);
            $currentDate = Carbon::now();
            $daysDifference = $investmentDate->diffInDays($currentDate);
            
            // Get money back period from schema (default 15 days)
            $moneyBackPeriod = $investment->schema->money_back_period ?? 15;
            
            // Calculate deduction based on time
            if ($daysDifference <= $moneyBackPeriod) {
                // Within 15 days: 20% deduction
                $deductionPercentage = 20;
                $deductionReason = "Early cancellation within {$moneyBackPeriod} days";
            } else {
                // After 15 days: Full cashback (0% deduction)
                $deductionPercentage = 0;
                $deductionReason = "Cancellation after {$moneyBackPeriod} days - Full cashback";
            }
            
            $deductionAmount = ($investment->invest_amount * $deductionPercentage) / 100;
            $refundAmount = $investment->invest_amount - $deductionAmount;
            
            \Log::info('NEW Cancellation calculation', [
                'investment_amount' => $investment->invest_amount,
                'days_since_investment' => $daysDifference,
                'money_back_period' => $moneyBackPeriod,
                'deduction_percentage' => $deductionPercentage,
                'deduction_amount' => $deductionAmount,
                'refund_amount' => $refundAmount,
                'deduction_reason' => $deductionReason
            ]);

            // Update investment (only columns that exist)
            $updateData = [
                'status' => InvestStatus::Canceled,
            ];
            if (Schema::hasColumn('invests', 'is_cancelled')) {
                $updateData['is_cancelled'] = true;
            }
            if (Schema::hasColumn('invests', 'cancelled_at')) {
                $updateData['cancelled_at'] = Carbon::now();
            }
            if (Schema::hasColumn('invests', 'cancellation_reason')) {
                $updateData['cancellation_reason'] = $request->reason;
            }
            if (Schema::hasColumn('invests', 'cancellation_deduction')) {
                $updateData['cancellation_deduction'] = $deductionAmount;
            }
            if (Schema::hasColumn('invests', 'refund_amount')) {
                $updateData['refund_amount'] = $refundAmount;
            }
            $investment->update($updateData);

            // Refund amount to user balance
            // Refund amount to user balance (supports decimal types)
            $user->balance = (float)$user->balance + (float)$refundAmount;
            $user->save();
            
            \Log::info('User balance updated', [
                'user_id' => $user->id,
                'refund_amount' => $refundAmount,
                'new_balance' => $user->balance
            ]);

            // Create refund transaction
            \Log::info('Creating refund transaction', [
                'refund_amount' => $refundAmount,
                'deduction_reason' => $deductionReason
            ]);
            
            $planName = $investment->schema ? $investment->schema->name : 'Unknown Plan';
            $refundTransaction = Transaction::create([
                'user_id' => $user->id,
                'amount' => $refundAmount,
                'charge' => 0,
                'final_amount' => $refundAmount,
                'type' => TxnType::Refund->value,
                'status' => TxnStatus::Success->value,
                'tnx' => 'REFUND_' . time(),
                'details' => [
                    'note' => $deductionReason,
                    'plan' => $planName,
                    'investment_id' => $investment->id,
                ],
                'description' => 'Investment Cancellation Refund',
            ]);

            \Log::info('Refund transaction created successfully', [
                'transaction_id' => $refundTransaction->id
            ]);

            // Create deduction transaction (only if there's a deduction)
            if ($deductionAmount > 0) {
                \Log::info('Creating deduction transaction', [
                    'deduction_amount' => $deductionAmount
                ]);
                
                $deductionTransaction = Transaction::create([
                    'user_id' => $user->id,
                    'amount' => $deductionAmount,
                    'charge' => 0,
                    'final_amount' => $deductionAmount,
                    'type' => TxnType::Charge->value,
                    'status' => TxnStatus::Success->value,
                    'tnx' => 'DEDUCTION_' . time(),
                    'details' => [
                        'note' => $deductionReason,
                        'plan' => $planName,
                        'investment_id' => $investment->id,
                    ],
                    'description' => 'Investment Cancellation Deduction',
                ]);

                \Log::info('Deduction transaction created successfully', [
                    'transaction_id' => $deductionTransaction->id
                ]);
            }

            DB::commit();

            \Log::info('Investment cancellation completed successfully', [
                'investment_id' => $investment->id,
                'refund_amount' => $refundAmount,
                'deduction_amount' => $deductionAmount
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Investment cancelled successfully',
                'refund_amount' => $refundAmount,
                'deduction_amount' => $deductionAmount,
                'days_since_investment' => $daysDifference,
                'deduction_reason' => $deductionReason
            ]);

        } catch (\Throwable $e) {
            // Safe rollback
            try { DB::rollBack(); } catch (\Throwable $t) {}

            \Log::error('Investment cancellation failed', [
                'investment_id' => $request->input('investment_id'),
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to cancel investment: ' . $e->getMessage(),
                'error_details' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ], 500);
        }
    }
}