<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\TxnType;
use App\Http\Controllers\Controller;
use App\Models\ReferralLink;
use App\Models\ReferralProgram;
use App\Models\LevelReferral;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReferralController extends Controller
{
    public function referral()
    {
        // Stop if referrals are disabled in settings
        if (!setting('sign_up_referral', 'permission')) {
            abort(404);
        }

        $user = auth()->user();
        $referralType = setting('site_referral', 'global'); // 'level' or 'global'
        $referralLevelSetting = setting('referral_level', 'global');

        // Fetch the referral link
        $getReferral = ReferralLink::getReferral($user, ReferralProgram::first());

        // ✅ FIXED: Generate proper referral links
        $getReferral->link = route('register.referral', $user->username);
        $getReferral->direct_link = route('register') . '?invite=' . $getReferral->code;
        // ✅ ADDED: Add referral count to getReferral object
        $getReferral->referralCount = $user->getTotalReferralCount();

        // Handle referral transactions grouped appropriately
        if ($referralType === 'level') {
            $referrals = Transaction::where('user_id', $user->id)
                ->whereNotNull('target_type')
                ->where('is_level', 1)
                ->get()
                ->groupBy('level');
        } else {
            $referrals = Transaction::where('user_id', $user->id)
                ->whereNotNull('target_type')
                ->get()
                ->groupBy('target');
        }

        // ✅ CHANGED: Show referral users instead of transactions in general tab
        $generalReferrals = $user->referrals()
            ->select('users.*')
            ->addSelect([
                'total_deposit' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type = "deposit")'),
                'total_invest' => DB::raw('(SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id AND status = "ongoing")'),
                'total_invest_all' => DB::raw('(SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id)'),
                'join_date' => DB::raw('users.created_at')
            ])
            ->latest()
            ->paginate(8)
            ->through(function ($user) {
                // Convert join_date to Carbon instance
                $user->join_date = \Carbon\Carbon::parse($user->join_date);
                return $user;
            });

        $totalReferralProfit = $user->totalReferralProfit();
        $currency = setting('site_currency', 'global');
        $level = LevelReferral::max('the_order');

        // ✅ FIXED: Use direct ref_id count for accurate referral counting
        $referralCount = \DB::table('users')->where('ref_id', $user->id)->count();
        
        // ✅ FIXED: Get referral users with proper data
        $referralUsers = $user->referrals()
            ->select('users.*')
            ->addSelect([
                'balance' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type = "deposit")'),
                'pending_balance' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "pending" AND type = "deposit")'),
                'second_level_count' => DB::raw('(SELECT COUNT(*) FROM users as u2 WHERE u2.ref_id IN (SELECT id FROM users WHERE ref_id = users.id))'),
                'currentSale' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type IN ("deposit", "investment"))'),
                'requiredSale' => DB::raw('0'),
                'pendingAmount' => DB::raw('0'),
                'requirementMet' => DB::raw('false'),
                'referCount' => DB::raw('(SELECT COUNT(*) FROM users WHERE ref_id = users.id)'),
                'join_date' => DB::raw('users.created_at'),
                // ✅ ADDED: Investment data
                'total_invest' => DB::raw('(SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id AND status = "ongoing")'),
                'total_invest_all' => DB::raw('(SELECT COALESCE(SUM(invest_amount), 0) FROM invests WHERE user_id = users.id)'),
                'total_profit' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type = "profit")'),
                'total_deposit' => DB::raw('(SELECT COALESCE(SUM(amount), 0) FROM transactions WHERE user_id = users.id AND status = "success" AND type = "deposit")')
            ])
            ->get()
            ->map(function ($user) {
                // Convert join_date to Carbon instance
                $user->join_date = \Carbon\Carbon::parse($user->join_date);
                return $user;
            });
        
        // ✅ FIXED: Set referral data without debug information
        $getReferral->referralCount = $referralCount;
        $getReferral->referralUsers = $referralUsers;

        return view('frontend.shahdeveloper.referral.index', compact(
            'getReferral',
            'referralLevelSetting',
            'referrals',
            'generalReferrals',
            'totalReferralProfit',
            'currency',
            'level',
            'referralCount',
            'referralUsers'
        ));
    }

    public function referralUsers()
    {
        $user = auth()->user();

        $referralUsers = $user->getReferralUsers();
        $totalReferrals = $user->referrals()->count();
        $totalEarnings = $user->totalReferralProfit();
        $currency = setting('site_currency', 'global');
        $initialPendingAmount = 900; // Static pending amount, customizable

        return view('frontend::referral.users', compact(
            'referralUsers',
            'totalReferrals',
            'totalEarnings',
            'currency',
            'initialPendingAmount'
        ));
    }
}
