<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\RoiSchedule;
use App\Models\Invest;
use App\Enums\TxnStatus;
use App\Enums\TxnType;
use Txn;
use Illuminate\Http\Request;
use Auth;
use Carbon\Carbon;

class RoiCollectionController extends Controller
{
    /**
     * Display user's ROI collection page
     */
    public function index()
    {
        $user = Auth::user();
        
        // Get user's active investments with their ROI schedules
        $userInvestments = Invest::where('user_id', $user->id)
            ->where('status', 'ongoing')
            ->with(['schema', 'roiSchedules' => function($query) {
                $query->orderBy('created_at', 'desc');
            }])
            ->get();

        // Get ready ROI schedules for quick actions
        $readySchedules = RoiSchedule::where('user_id', $user->id)
            ->where('status', 'ready')
            ->with(['invest.schema'])
            ->get();

        // Calculate statistics
        $totalPending = RoiSchedule::where('user_id', $user->id)
            ->where('status', 'pending')
            ->sum('roi_amount');

        $totalReady = RoiSchedule::where('user_id', $user->id)
            ->where('status', 'ready')
            ->sum('roi_amount');

        $totalCollected = RoiSchedule::where('user_id', $user->id)
            ->where('status', 'sent')
            ->sum('roi_amount');

        $statistics = [
            'pending' => $totalPending,
            'ready' => $totalReady,
            'collected' => $totalCollected,
            'total' => $totalPending + $totalReady + $totalCollected
        ];

        return view('frontend.shahdeveloper.user.collect-roi', compact(
            'userInvestments',
            'readySchedules',
            'statistics'
        ));
    }

    /**
     * Collect ready ROI
     */
    public function collectRoi(Request $request)
    {
        $request->validate([
            'schedule_id' => 'required|exists:roi_schedules,id'
        ]);

        $user = Auth::user();
        $schedule = RoiSchedule::where('id', $request->schedule_id)
            ->where('user_id', $user->id)
            ->where('status', 'ready')
            ->first();

        if (!$schedule) {
            // Check if the schedule exists but is pending and should be ready
            $schedule = RoiSchedule::where('id', $request->schedule_id)
                ->where('user_id', $user->id)
                ->where('status', 'pending')
                ->where('roi_end_time', '<=', now())
                ->first();
                
            if ($schedule) {
                // Mark as ready
                $schedule->update([
                    'status' => 'ready',
                    'roi_ready_time' => now()
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'ROI schedule not found or not ready for collection'
                ], 404);
            }
        }

        try {
            $roiAmount = $schedule->roi_amount;
            
            // Debug: Log before balance update
            \Log::info("ROI Collection Debug - User ID: {$user->id}, Username: {$user->username}, Current Balance: {$user->profit_balance}, ROI Amount: {$roiAmount}");
            \Log::info("ROI Collection Debug - Schedule ID: {$schedule->id}, Status: {$schedule->status}");

            // Add ROI to user's profit balance
            $result = $user->increment('profit_balance', $roiAmount);
            \Log::info("ROI Collection Debug - Increment result: " . ($result ? 'true' : 'false'));
            
            // Debug: Log after balance update
            $user->refresh();
            \Log::info("ROI Collection Debug - After Update, New Balance: {$user->profit_balance}");
            
            // Double check with direct query
            $directBalance = \DB::table('users')->where('id', $user->id)->value('profit_balance');
            \Log::info("ROI Collection Debug - Direct DB query balance: {$directBalance}");

            // Create transaction record
            Txn::new(
                $roiAmount,
                0, // No charge
                $roiAmount,
                'system',
                "ROI Collection from {$schedule->invest->schema->name} Plan - Cycle {$schedule->roi_cycle}",
                TxnType::Interest,
                TxnStatus::Success,
                null,
                null,
                $user->id
            );

            // Mark ROI as sent
            $schedule->update([
                'status' => 'sent',
                'roi_sent_time' => now()
            ]);

            // Create next ROI cycle if not completed
            if ($schedule->roi_cycle < $schedule->total_cycles) {
                $nextCycle = $schedule->roi_cycle + 1;
                
                // Get ROI period from schema settings (in hours)
                $roiPeriodHours = $schedule->invest->schema->roi_period ?? 24; // Default 24 hours
                
                RoiSchedule::create([
                    'invest_id' => $schedule->invest_id,
                    'user_id' => $schedule->user_id,
                    'schema_id' => $schedule->schema_id,
                    'invest_amount' => $schedule->invest_amount,
                    'roi_percentage' => $schedule->roi_percentage,
                    'roi_amount' => $schedule->roi_amount,
                    'roi_start_time' => now(),
                    'roi_end_time' => now()->addHours($roiPeriodHours),
                    'status' => 'pending',
                    'roi_cycle' => $nextCycle,
                    'total_cycles' => $schedule->total_cycles
                ]);
            } else {
                // Mark investment as completed if all cycles are done
                $schedule->invest->update(['status' => 'completed']);
            }

            return response()->json([
                'success' => true,
                'message' => 'ROI collected successfully!',
                'amount' => $roiAmount,
                'new_balance' => $user->fresh()->profit_balance
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error collecting ROI: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Collect all ready ROI at once
     */
    public function collectAllRoi()
    {
        $user = Auth::user();
        
        $readySchedules = RoiSchedule::where('user_id', $user->id)
            ->where('status', 'ready')
            ->get();

        if ($readySchedules->count() === 0) {
            // Check if there are any pending schedules that should be ready
            $pendingSchedules = RoiSchedule::where('user_id', $user->id)
                ->where('status', 'pending')
                ->where('roi_end_time', '<=', now())
                ->get();
                
            if ($pendingSchedules->count() > 0) {
                // Mark them as ready
                foreach ($pendingSchedules as $schedule) {
                    $schedule->update([
                        'status' => 'ready',
                        'roi_ready_time' => now()
                    ]);
                }
                
                // Get the ready schedules again
                $readySchedules = RoiSchedule::where('user_id', $user->id)
                    ->where('status', 'ready')
                    ->get();
            }
            
            if ($readySchedules->count() === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No ready ROI found for collection. Please wait for your ROI to be ready.'
                ], 404);
            }
        }

        $totalAmount = 0;
        $processedCount = 0;

        try {
            foreach ($readySchedules as $schedule) {
                $roiAmount = $schedule->roi_amount;
                $totalAmount += $roiAmount;
                
                // Debug: Log before balance update
                \Log::info("ROI Collection All Debug - User ID: {$user->id}, Username: {$user->username}, Current Balance: {$user->profit_balance}, ROI Amount: {$roiAmount}");
                \Log::info("ROI Collection All Debug - Schedule ID: {$schedule->id}, Status: {$schedule->status}");
                
                // Add ROI to user's profit balance
                $result = $user->increment('profit_balance', $roiAmount);
                \Log::info("ROI Collection All Debug - Increment result: " . ($result ? 'true' : 'false'));
                
                // Debug: Log after balance update
                $user->refresh();
                \Log::info("ROI Collection All Debug - After Update, New Balance: {$user->profit_balance}");
                
                // Double check with direct query
                $directBalance = \DB::table('users')->where('id', $user->id)->value('profit_balance');
                \Log::info("ROI Collection All Debug - Direct DB query balance: {$directBalance}");

                // Create transaction record
                Txn::new(
                    $roiAmount,
                    0, // No charge
                    $roiAmount,
                    'system',
                    "ROI Collection from {$schedule->invest->schema->name} Plan - Cycle {$schedule->roi_cycle}",
                    TxnType::Interest,
                    TxnStatus::Success,
                    null,
                    null,
                    $user->id
                );

                // Mark ROI as sent
                $schedule->update([
                    'status' => 'sent',
                    'roi_sent_time' => now()
                ]);

                // Create next ROI cycle if not completed
                if ($schedule->roi_cycle < $schedule->total_cycles) {
                    $nextCycle = $schedule->roi_cycle + 1;
                    
                    // Get ROI period from schema settings (in hours)
                    $roiPeriodHours = $schedule->invest->schema->roi_period ?? 24; // Default 24 hours
                    
                    RoiSchedule::create([
                        'invest_id' => $schedule->invest_id,
                        'user_id' => $schedule->user_id,
                        'schema_id' => $schedule->schema_id,
                        'invest_amount' => $schedule->invest_amount,
                        'roi_percentage' => $schedule->roi_percentage,
                        'roi_amount' => $schedule->roi_amount,
                        'roi_start_time' => now(),
                        'roi_end_time' => now()->addHours($roiPeriodHours),
                        'status' => 'pending',
                        'roi_cycle' => $nextCycle,
                        'total_cycles' => $schedule->total_cycles
                    ]);
                } else {
                    // Mark investment as completed if all cycles are done
                    $schedule->invest->update(['status' => 'completed']);
                }

                $processedCount++;
            }

            return response()->json([
                'success' => true,
                'message' => "Successfully collected {$processedCount} ROI schedules!",
                'total_amount' => $totalAmount,
                'processed_count' => $processedCount,
                'new_balance' => $user->fresh()->profit_balance
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error collecting ROI: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get ROI statistics for dashboard
     */
    public function getRoiStats()
    {
        $user = Auth::user();
        
        $stats = [
            'pending' => RoiSchedule::where('user_id', $user->id)
                ->where('status', 'pending')
                ->sum('roi_amount'),
            'ready' => RoiSchedule::where('user_id', $user->id)
                ->where('status', 'ready')
                ->sum('roi_amount'),
            'collected' => RoiSchedule::where('user_id', $user->id)
                ->where('status', 'sent')
                ->sum('roi_amount'),
        ];

        $stats['total'] = $stats['pending'] + $stats['ready'] + $stats['collected'];

        return response()->json($stats);
    }

    public function getReadySchedule($investmentId)
    {
        $user = Auth::user();
        
        // Find ready ROI schedule for this specific investment
        $schedule = RoiSchedule::where('user_id', $user->id)
            ->where('invest_id', $investmentId)
            ->where('status', 'ready')
            ->first();
            
        if (!$schedule) {
            // Check if there's a pending schedule that should be ready
            $schedule = RoiSchedule::where('user_id', $user->id)
                ->where('invest_id', $investmentId)
                ->where('status', 'pending')
                ->where('roi_end_time', '<=', now())
                ->first();
                
            if ($schedule) {
                // Mark as ready
                $schedule->update([
                    'status' => 'ready',
                    'roi_ready_time' => now()
                ]);
            }
        }
        
        if ($schedule) {
            return response()->json([
                'success' => true,
                'schedule_id' => $schedule->id,
                'amount' => $schedule->roi_amount,
                'cycle' => $schedule->roi_cycle
            ]);
        }
        
        return response()->json([
            'success' => false,
            'message' => 'No ready ROI found for this investment'
        ], 404);
    }
}
