<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\WithdrawMethod;
use App\Traits\NotifyTrait;
use App\Enums\TxnType;
use App\Enums\TxnStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class SimpleWithdrawController extends Controller
{
    use NotifyTrait;

    /**
     * Get withdraw methods from database
     */
    private function getWithdrawMethods()
    {
        // Only get methods that are active in admin panel
        $methods = WithdrawMethod::where('status', 1)->get();
        
        $formattedMethods = [
            'crypto' => [],
            'banking' => []
        ];
        
        foreach ($methods as $method) {
            // Handle icon - if it's an image path, use it; otherwise use emoji
            $icon = $method->icon ?? '💰';
            if (strpos($icon, '.png') !== false || strpos($icon, '.jpg') !== false || strpos($icon, '.jpeg') !== false) {
                // It's an image path, keep as is
                $icon = $icon;
            } else {
                // It's an emoji or text, use as is
                $icon = $icon;
            }
            
            $methodData = [
                'id' => $method->id,
                'name' => $method->name,
                'icon' => $icon,
                'fields' => json_decode($method->fields ?? '[]', true) ?: [],
                'min_amount' => $method->min_amount ?? 10,
                'max_amount' => $method->max_amount ?? 10000,
                'fee_percentage' => $method->charge ?? 0,
                'rate' => $method->rate ?? 1,
                'currency' => $method->currency ?? 'USD'
            ];
            
            // Categorize methods based on type field or name
            $methodName = strtolower($method->name);
            $isCrypto = false;
            
            // Check if method has type field
            if (isset($method->type) && $method->type === 'crypto') {
                $isCrypto = true;
            } else {
                // Check by name patterns
                $cryptoKeywords = ['solana', 'arbitrum', 'bnb', 'usdt', 'bitcoin', 'ethereum', 'tron', 'trc20', 'erc20', 'smart chain'];
                foreach ($cryptoKeywords as $keyword) {
                    if (strpos($methodName, $keyword) !== false) {
                        $isCrypto = true;
                        break;
                    }
                }
            }
            
            if ($isCrypto) {
                $formattedMethods['crypto'][strtolower(str_replace([' ', '(', ')', '-'], ['_', '', '', '_'], $method->name))] = $methodData;
            } else {
                $formattedMethods['banking'][strtolower(str_replace([' ', '(', ')', '-'], ['_', '', '', '_'], $method->name))] = $methodData;
            }
        }
        
        // If no methods in database, return default methods
        if (empty($formattedMethods['crypto']) && empty($formattedMethods['banking'])) {
            return [
                'crypto' => [
                    'solana' => [
                        'id' => 'default_solana',
                        'name' => 'Solana (SOL)',
                        'icon' => '💎',
                        'fields' => ['wallet_address'],
                        'min_amount' => 10,
                        'max_amount' => 10000,
                        'fee_percentage' => 0,
                        'rate' => 1,
                        'currency' => 'USD'
                    ]
                ],
                'banking' => [
                    'easypaisa' => [
                        'id' => 'default_easypaisa',
                        'name' => 'EasyPaisa',
                        'icon' => '📱',
                        'fields' => ['account_number', 'account_title'],
                        'min_amount' => 10,
                        'max_amount' => 50000,
                        'fee_percentage' => 0,
                        'rate' => 1,
                        'currency' => 'USD'
                    ]
                ]
            ];
        }
        
        return $formattedMethods;
    }

    /**
     * Show withdraw form
     */
    public function index()
    {
        // Check global withdraw permission
        if (!setting('user_withdraw', 'permission')) {
            notify()->error(__('Withdraw is currently disabled by admin.'), 'Withdraw Disabled');
            return redirect()->back();
        }

        // Check user's withdraw status
        if (!Auth::user()->withdraw_status) {
            notify()->error(__('Your withdraw has been disabled by admin. Please contact support for assistance.'), 'Withdraw Disabled');
            return redirect()->route('user.ticket.new');
        }

       

        $methods = $this->getWithdrawMethods();
        $user = Auth::user();
        $currency = setting('site_currency', 'global');
        $currencySymbol = setting('currency_symbol', 'global');
        
        return view('frontend.shahdeveloper.user.withdraw.simple', compact('methods', 'user', 'currency', 'currencySymbol'));
    }

    /**
     * Process withdraw request
     */
    public function store(Request $request)
    {
        \Log::info('Withdraw request started', $request->all());
        try {
            // Check global withdraw permission
            try {
                $withdrawEnabled = setting('user_withdraw', 'permission');
                if (!$withdrawEnabled) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Withdraw is currently disabled by admin.'
                    ]);
                }
            } catch (\Exception $e) {
                // If setting function fails, allow withdraw by default
                \Log::warning('Setting function failed in withdraw: ' . $e->getMessage());
            }

        // Check user's withdraw status
        if (!Auth::user()->withdraw_status) {
            return response()->json([
                'success' => false,
                'message' => 'Your withdraw has been disabled by admin. Please contact support for assistance.'
            ]);
        }

        // Profile Completion Check
        try {
            if (!Auth::user()->isProfileComplete()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Your profile must be 80% complete to withdraw funds. Please complete your profile first.'
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('Profile completion check failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error checking profile completion. Please try again.'
            ]);
        }

        $input = $request->all();
        $methods = $this->getWithdrawMethods();
        
        // Debug: Log received data
        \Log::info('Withdraw form data received:', $input);
        
        // Basic validation
        $validator = Validator::make($input, [
            'method_type' => 'required|in:crypto,banking',
            'crypto_method' => 'required_if:method_type,crypto',
            'banking_method' => 'required_if:method_type,banking',
            'amount' => 'required|numeric|min:1',
            'wallet' => 'required|in:main,profit'
        ]);

        if ($validator->fails()) {
            \Log::error('Validation failed:', $validator->errors()->toArray());
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
                'errors' => $validator->errors()->toArray()
            ]);
        }

        $methodType = $input['method_type'];
        $methodKey = $input['method_type'] === 'crypto' ? $input['crypto_method'] : $input['banking_method'];
        
        // Check if method exists
        if (!isset($methods[$methodType][$methodKey])) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid withdraw method selected'
            ]);
        }

        $methodInfo = $methods[$methodType][$methodKey];
        $amount = (float)$input['amount'];
        
        // Debug: Log method info
        \Log::info('Method info:', $methodInfo);
        \Log::info('Fields in method info:', $methodInfo['fields'] ?? 'NOT SET');

        // Validate amount range
        if ($amount < $methodInfo['min_amount'] || $amount > $methodInfo['max_amount']) {
            $currencySymbol = setting('currency_symbol', 'global');
            $message = 'Amount must be between ' . $currencySymbol . $methodInfo['min_amount'] . ' and ' . $currencySymbol . $methodInfo['max_amount'];
            return response()->json([
                'success' => false,
                'message' => $message
            ]);
        }

        // Dynamic field validation based on method
        $fieldRules = [];
        if (isset($methodInfo['fields']) && is_array($methodInfo['fields'])) {
            foreach ($methodInfo['fields'] as $field) {
                $fieldRules[$field] = 'required';
            }
        }

        if (!empty($fieldRules)) {
            $fieldValidator = Validator::make($input, $fieldRules);
            if ($fieldValidator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $fieldValidator->errors()->first()
                ]);
            }
        }

        $user = Auth::user();
        
        // Calculate fees
        $feeAmount = ($amount * $methodInfo['fee_percentage']) / 100;
        $totalAmount = $amount + $feeAmount;

        // Check user balance (including pending withdraw requests) based on selected wallet
        $walletType = $input['wallet']; // 'main' or 'profit'
        
        $pendingWithdraws = Transaction::where('user_id', $user->id)
            ->where('type', TxnType::Withdraw)
            ->where('status', TxnStatus::Pending)
            ->where('details->wallet', $walletType) // Check pending withdrawals from same wallet
            ->sum('amount');
            
        if ($walletType === 'main') {
            $availableBalance = $user->balance - $pendingWithdraws;
        } else {
            $availableBalance = $user->profit_balance - $pendingWithdraws;
        }
        
        if ($availableBalance < $totalAmount) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient balance. You have pending withdraw requests.'
            ]);
        }

        // Collect account details
        $fields = isset($methodInfo['fields']) && is_array($methodInfo['fields']) ? $methodInfo['fields'] : [];
        $accountDetails = $this->collectAccountDetails($input, $fields);
        
        // Use database transaction for data integrity
        \DB::beginTransaction();
        try {
            // Deduct amount from the correct wallet based on user selection
            $walletType = $input['wallet']; // 'main' or 'profit'
            
            if ($walletType === 'main') {
                // Deduct from main balance
                $newBalance = $user->balance - $totalAmount;
                $updated = \DB::table('users')->where('id', $user->id)->update(['balance' => $newBalance]);
                
                \Log::info('Main balance deducted', [
                    'user_id' => $user->id,
                    'old_balance' => $user->balance,
                    'amount_deducted' => $totalAmount,
                    'new_balance' => $newBalance,
                    'rows_affected' => $updated
                ]);
            } else {
                // Deduct from profit balance
                $newProfitBalance = $user->profit_balance - $totalAmount;
                $updated = \DB::table('users')->where('id', $user->id)->update(['profit_balance' => $newProfitBalance]);
                
                \Log::info('Profit balance deducted', [
                    'user_id' => $user->id,
                    'old_profit_balance' => $user->profit_balance,
                    'amount_deducted' => $totalAmount,
                    'new_profit_balance' => $newProfitBalance,
                    'rows_affected' => $updated
                ]);
            }

            // Create transaction record with pending status for admin approval
            $transaction = new Transaction();
            $transaction->user_id = $user->id;
            $transaction->amount = $totalAmount;
            $transaction->charge = $feeAmount;
            $transaction->final_amount = $amount;
            $transaction->tnx = 'TXN' . time() . rand(1000, 9999);
            $transaction->type = TxnType::Withdraw;
            $transaction->status = TxnStatus::Pending;
            $transaction->method = 'SIMPLE-' . strtoupper($methodKey);
            $transaction->currency = setting('site_currency', 'global');
            $transaction->details = json_encode($accountDetails);
            $transaction->save();
            
            \Log::info('Transaction created successfully', [
                'transaction_id' => $transaction->id,
                'type' => $transaction->type->value,
                'status' => $transaction->status->value,
                'method' => $transaction->method,
                'amount' => $transaction->amount
            ]);
            
            \DB::commit();
        } catch (\Exception $e) {
            \DB::rollback();
            \Log::error('Transaction creation failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create transaction record. Please try again.'
            ]);
        }

        // Send notification to admin about new withdraw request
        $shortcodes = [
            '[[full_name]]' => $user->full_name,
            '[[email]]' => $user->email,
            '[[amount]]' => $amount,
            '[[method]]' => $methodInfo['name'],
            '[[site_title]]' => setting('site_title', 'global'),
            '[[site_url]]' => route('home'),
        ];

        // Send notification to admin about new withdraw request
        try {
            if (method_exists($this, 'mailNotify')) {
                $this->mailNotify(setting('site_email', 'global'), 'withdraw_request', $shortcodes);
            }
            if (method_exists($this, 'pushNotify')) {
                $this->pushNotify('withdraw_request', $shortcodes, route('admin.withdraw.pending'), $user->id);
            }
        } catch (\Exception $e) {
            \Log::error('Notification error: ' . $e->getMessage());
        }

            // Return JSON response for AJAX
            return response()->json([
                'success' => true,
                'message' => 'Withdraw request submitted successfully! Admin will process it soon.',
                'redirect' => route('user.withdraw.simple.success')
            ]);
            
        } catch (\Exception $e) {
            \Log::error('Withdraw error: ' . $e->getMessage());
            \Log::error('Withdraw error trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Withdraw request failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Show withdraw history
     */
    public function history()
    {
        $user = Auth::user();
        $currency = setting('site_currency', 'global');
        $currencySymbol = setting('currency_symbol', 'global');
        
        $withdrawals = Transaction::where('user_id', $user->id)
            ->whereIn('type', [TxnType::Withdraw, 'withdraw_auto'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        
        return view('frontend.shahdeveloper.user.withdraw.history', compact('withdrawals', 'currency', 'currencySymbol'));
    }

    /**
     * Show withdraw success page
     */
    public function success()
    {
        return view('frontend.shahdeveloper.user.withdraw.success');
    }

    /**
     * Collect account details based on method requirements
     */
    private function collectAccountDetails($input, $fields)
    {
        $details = [];
        
        // Add method type and selected method
        $details['method_type'] = $input['method_type'];
        if ($input['method_type'] === 'crypto') {
            $details['crypto_method'] = $input['crypto_method'];
        } else {
            $details['banking_method'] = $input['banking_method'];
        }
        
        // Add wallet selection
        $details['wallet'] = $input['wallet'];
        
        // Add crypto details
        if (isset($input['wallet_address'])) {
            $details['wallet_address'] = $input['wallet_address'];
        }
        if (isset($input['network_type'])) {
            $details['network_type'] = $input['network_type'];
        }
        
        // Add banking details
        if (isset($input['account_title'])) {
            $details['account_title'] = $input['account_title'];
        }
        if (isset($input['account_number'])) {
            $details['account_number'] = $input['account_number'];
        }
        if (isset($input['bank_name'])) {
            $details['bank_name'] = $input['bank_name'];
        }
        if (isset($input['branch_code'])) {
            $details['branch_code'] = $input['branch_code'];
        }
        
        // Add mobile payment details
        if (isset($input['mobile_number'])) {
            $details['mobile_number'] = $input['mobile_number'];
        }
        if (isset($input['account_name'])) {
            $details['account_name'] = $input['account_name'];
        }
        
        return $details;
    }
}