<?php

namespace App\Http\Controllers\Frontend;

use App\Enums\TxnStatus;
use App\Enums\TxnType;
use App\Http\Controllers\Controller;
use App\Models\Transaction;
use App\Models\WithdrawRequest;
use Illuminate\Support\Facades\Auth;
use App\Models\WithdrawAccount;
use App\Models\WithdrawalSchedule;
use App\Models\WithdrawMethod;
use App\Models\WithdrawOtp;
use App\Traits\ImageUpload;
use App\Traits\NotifyTrait;
use App\Traits\Payment;
use Carbon\Carbon;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Session;
use Txn;
use Validator;

class WithdrawController extends Controller
{
    use ImageUpload, NotifyTrait, Payment;

    /**
     * Display a listing of the resource.
     *
     * @return Application|Factory|View
     */
    public function index()
    {
        $accounts = WithdrawAccount::where('user_id', auth()->id())->get();

        return view('frontend.shahdeveloper.user.withdraw.account.index', compact('accounts'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return RedirectResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'withdraw_method_id' => 'required',
            'method_name' => 'required',
            'credentials' => 'required',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        $input = $request->all();
        $credentials = $input['credentials'];

        foreach ($credentials as $key => $value) {
            if (isset($value['value']) && is_file($value['value'])) {
                $credentials[$key]['value'] = self::imageUploadTrait($value['value']);
            }
        }

        $data = [
            'user_id' => auth()->id(),
            'withdraw_method_id' => $input['withdraw_method_id'],
            'method_name' => $input['method_name'],
            'credentials' => json_encode($credentials),
        ];

        WithdrawAccount::create($data);
        notify()->success('Successfully Withdraw Account Created', 'success');

        return redirect()->route('user.withdraw.account.index');
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Application|Factory|View
     */
    public function create()
    {
        $withdrawMethods = WithdrawMethod::where('status', true)->get();

        return view('frontend.shahdeveloper.user.withdraw.account.create', compact('withdrawMethods'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return Application|Factory|View
     */
    public function edit($id)
    {
        $withdrawMethods = WithdrawMethod::all();
        $withdrawAccount = WithdrawAccount::find($id);

        return view('frontend.shahdeveloper.user.withdraw.account.edit', compact('withdrawMethods', 'withdrawAccount'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param int $id
     * @return RedirectResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'withdraw_method_id' => 'required',
            'method_name' => 'required',
            'credentials' => 'required',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        $input = $request->all();
        $withdrawAccount = WithdrawAccount::find($id);
        $oldCredentials = json_decode($withdrawAccount->credentials, true);
        $credentials = $input['credentials'];

        foreach ($credentials as $key => $value) {
            if (!isset($value['value'])) {
                $credentials[$key]['value'] = data_get($oldCredentials[$key],'value');
            }

            if (isset($value['value']) && is_file($value['value'])) {
                $credentials[$key]['value'] = self::imageUploadTrait($value['value'], data_get($oldCredentials[$key],'value'));
            }
        }

        $data = [
            'user_id' => auth()->id(),
            'withdraw_method_id' => $input['withdraw_method_id'],
            'method_name' => $input['method_name'],
            'credentials' => json_encode($credentials),
        ];

        $withdrawAccount->update($data);
        notify()->success('Successfully Withdraw Account Updated', 'success');

        return redirect()->route('user.withdraw.account.index');
    }

    /**
     * @return string
     */
    public function withdrawMethod($id)
    {
        $withdrawMethod = WithdrawMethod::find($id);

        if ($withdrawMethod) {
            return view('frontend::withdraw.include.__account', compact('withdrawMethod'))->render();
        }

        return '';
    }

    /**
     * @return array
     */
    public function details($accountId, int $amount = 0)
    {
        $withdrawAccount = WithdrawAccount::find($accountId);
        $credentials = json_decode($withdrawAccount->credentials, true);
        $currency = setting('site_currency', 'global');
        $method = $withdrawAccount->method;
        $charge = $method->charge;
        $name = $withdrawAccount->method_name;
        $processingTime = (int)$method->required_time > 0 ? 'Processing Time: ' . $withdrawAccount->method->required_time . $withdrawAccount->method->required_time_format : 'This Is Automatic Method';

        $info = [
            'name' => $name,
            'charge' => $charge,
            'charge_type' => $withdrawAccount->method->charge_type,
            'range' => 'Minimum ' . $method->min_withdraw . ' ' . $currency . ' and ' . 'Maximum ' . $method->max_withdraw . ' ' . $currency,
            'processing_time' => $processingTime,
            'rate' => $method->rate,
            'pay_currency' => $method->currency
        ];

        if ($withdrawAccount->method->charge_type != 'fixed') {
            $charge = ($charge / 100) * $amount;
        }
        
        $conversionRate = $method->currency != $currency ? $method->rate . ' ' . $method->currency : null;
        $html = view('frontend::withdraw.include.__details', compact('credentials', 'name', 'charge', 'conversionRate'))->render();

        return [
            'html' => $html,
            'info' => $info,
        ];
    }

    /**
     * Process withdrawal request with OTP verification
     *
     * @return RedirectResponse
     */
    public function withdrawNow(Request $request)
    {
        // Check global withdraw permission
        if (!setting('user_withdraw', 'permission')) {
            notify()->error(__('Withdraw is currently disabled by admin.'), 'Withdraw Disabled');
            return redirect()->back();
        }

        // Check user's withdraw status
        if (!\Auth::user()->withdraw_status) {
            notify()->error(__('Your withdraw has been disabled by admin. Please contact support for assistance.'), 'Withdraw Disabled');
            return redirect()->route('user.ticket.new');
        }

        // KYC Verification Check
        if (setting('kyc_verification', 'permission') && \Auth::user()->kyc != \App\Enums\KYCStatus::Verified->value) {
            notify()->error(__('KYC verification is required to withdraw funds. Please complete your KYC verification first.'), 'KYC Required');
            return redirect()->route('user.kyc');
        }

        $withdrawOffDays = WithdrawalSchedule::where('status', 0)->pluck('name')->toArray();
        $date = Carbon::now();
        $today = $date->format('l');

        if (in_array($today, $withdrawOffDays)) {
            abort('403', __('Today is the off day of withdraw'));
        }

        $validator = Validator::make($request->all(), [
            'amount' => ['required', 'regex:/^[0-9]+(\.[0-9][0-9]?)?$/'],
            'withdraw_account' => 'required',
        ]);

        if ($validator->fails()) {
            notify()->error($validator->errors()->first(), 'Error');
            return redirect()->back();
        }

        // Daily limit check
        $todayTransaction = Transaction::where('user_id', Auth::id())
            ->where('type', TxnType::Withdraw)
            ->orWhere('type', TxnType::WithdrawAuto)
            ->whereDate('created_at', Carbon::today())
            ->count();
            
        $dayLimit = (float)Setting('withdraw_day_limit', 'fee');
        if ($todayTransaction >= $dayLimit) {
            notify()->error(__('Today Withdraw limit has been reached'), 'Error');
            return redirect()->back();
        }

        $input = $request->all();
        $amount = (float)$input['amount'];

        $withdrawAccount = WithdrawAccount::find($input['withdraw_account']);
        $withdrawMethod = $withdrawAccount->method;

        if ($amount < $withdrawMethod->min_withdraw || $amount > $withdrawMethod->max_withdraw) {
            $currencySymbol = setting('currency_symbol', 'global');
            $message = 'Please Withdraw the Amount within the range ' . $currencySymbol . $withdrawMethod->min_withdraw . ' to ' . $currencySymbol . $withdrawMethod->max_withdraw;
            notify()->error($message, 'Error');
            return redirect()->back();
        }

        // Generate and store OTP
        $otp = rand(100000, 999999); // 6-digit numeric OTP
        
        $charge = $withdrawMethod->charge_type == 'percentage' ? (($withdrawMethod->charge / 100) * $amount) : $withdrawMethod->charge;

        $withdrawData = [
            'amount' => $amount,
            'withdraw_account' => $input['withdraw_account'],
            'charge' => $charge,
            'total_amount' => $amount + (float)$charge,
            'method_name' => $withdrawMethod->name,
            'credentials' => json_decode($withdrawAccount->credentials, true),
        ];

        WithdrawOtp::updateOrCreate(
            ['user_id' => auth()->id()],
            [
                'otp' => $otp,
                'transaction_data' => json_encode($withdrawData),
                'expires_at' => now()->addMinutes(15),
                'verified' => false
            ]
        );

        // Send OTP to user
        $this->sendWithdrawOtp(auth()->user(), $otp);

        notify()->success(__('OTP sent to your email address. Please verify to complete withdrawal.'), 'Success');
        return redirect()->route('user.withdraw.verify.otp');
    }

    /**
     * Show OTP verification form
     *
     * @return Application|Factory|View
     */
    public function showVerifyOtpForm()
    {
        return view('frontend.shahdeveloper.user.withdraw.verify_otp');
    }

    /**
     * Verify OTP and process withdrawal
     *
     * @return RedirectResponse
     */
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'otp' => 'required|digits:6'
        ]);

        $otpRecord = WithdrawOtp::where('user_id', auth()->id())
            ->where('otp', $request->otp)
            ->where('verified', false)
            ->where('expires_at', '>', now())
            ->first();

        if (!$otpRecord) {
            notify()->error(__('Invalid or expired OTP'), 'Error');
            return redirect()->back();
        }

        // Mark OTP as verified
        $otpRecord->update(['verified' => true]);

        // Process the withdrawal
        $withdrawData = json_decode($otpRecord->transaction_data, true);
        return $this->processWithdrawal($withdrawData);
    }

    /**
     * Process the withdrawal after OTP verification
     *
     * @param array $withdrawData
     * @return RedirectResponse
     */
    protected function processWithdrawal(array $withdrawData)
    {
        $user = Auth::user();
        $amount = (float)$withdrawData['amount'];
        $withdrawAccount = WithdrawAccount::find($withdrawData['withdraw_account']);
        $withdrawMethod = $withdrawAccount->method;
        $charge = (float)$withdrawData['charge'];
        $totalAmount = (float)$withdrawData['total_amount'];

        if ($user->balance < $totalAmount) {
            notify()->error(__('Insufficient Balance Your Main Wallet'), 'Error');
            return redirect()->back();
        }

        // Use database transaction to ensure atomicity
        \DB::beginTransaction();
        
        try {
            // Log balance before deduction
            \Log::info('Withdraw Balance Deduction', [
                'user_id' => $user->id,
                'balance_before' => $user->balance,
                'total_amount' => $totalAmount,
                'balance_after' => $user->balance - $totalAmount
            ]);

            // Use raw database query to ensure balance is deducted
            $newBalance = $user->balance - $totalAmount;
            
            // Direct database update to ensure reliable balance deduction
            $updated = DB::table('users')
                ->where('id', $user->id)
                ->update(['balance' => $newBalance]);
            
            Log::info('Balance updated using raw query in WithdrawController', [
                'user_id' => $user->id,
                'old_balance' => $user->balance,
                'new_balance' => $newBalance,
                'rows_affected' => $updated
            ]);
            
            // Refresh user and log balance after deduction
            $user->refresh();
            \Log::info('Withdraw Balance After Deduction', [
                'user_id' => $user->id,
                'balance_after_deduction' => $user->balance
            ]);
            $payAmount = $amount * $withdrawMethod->rate;
            $type = $withdrawMethod->type == 'auto' ? TxnType::WithdrawAuto : TxnType::Withdraw;

            $txnInfo = Txn::new(
                $amount,
                $charge,
                $totalAmount,
                $withdrawMethod->name,
                'Withdraw With ' . $withdrawAccount->method_name,
                $type,
                TxnStatus::Pending,
                $withdrawMethod->currency,
                $payAmount,
                $user->id,
                null,
                'User',
                $withdrawData['credentials']
            );

            // Commit the transaction
            \DB::commit();
            
            \Log::info('Withdraw Transaction Created Successfully', [
                'user_id' => $user->id,
                'transaction_id' => $txnInfo->id,
                'amount' => $amount,
                'total_amount' => $totalAmount,
                'final_balance' => $user->balance
            ]);

            if ($withdrawMethod->type == 'auto') {
                $gatewayCode = $withdrawMethod->gateway->gateway_code;
                return self::withdrawAutoGateway($gatewayCode, $txnInfo);
            }

            $symbol = setting('currency_symbol', 'global');
            $notify = [
                'card-header' => 'Withdraw Money',
                'title' => $symbol . $txnInfo->amount . ' Withdraw Request Successful',
                'p' => 'The Withdraw Request has been successfully sent',
                'strong' => 'Transaction ID: ' . $txnInfo->tnx,
                'action' => route('user.withdraw.view'),
                'a' => 'WITHDRAW REQUEST AGAIN',
                'view_name' => 'withdraw',
            ];
            
            Session::put('user_notify', $notify);
            
        } catch (\Exception $e) {
            // Rollback the transaction on error
            \DB::rollback();
            
            \Log::error('Withdraw Transaction Failed', [
                'user_id' => $user->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            notify()->error('Withdrawal failed. Please try again.', 'Error');
            return redirect()->back();
        }
        
        $shortcodes = [
            '[[full_name]]' => $txnInfo->user->full_name,
            '[[txn]]' => $txnInfo->tnx,
            '[[method_name]]' => $withdrawMethod->name,
            '[[withdraw_amount]]' => $txnInfo->amount . setting('site_currency', 'global'),
            '[[site_title]]' => setting('site_title', 'global'),
            '[[site_url]]' => route('home'),
        ];

        $this->mailNotify(setting('site_email', 'global'), 'withdraw_request', $shortcodes);
        $this->pushNotify('withdraw_request', $shortcodes, route('admin.withdraw.pending'), $user->id);
        $this->smsNotify('withdraw_request', $shortcodes, $user->phone);

        return redirect()->route('user.notify');
    }

    /**
     * Send OTP to user's email with enhanced logging and error handling
     *
     * @param \App\Models\User $user
     * @param string $otp
     * @return bool
     */
 protected function sendWithdrawOtp($user, $otp)
{
    try {
        $data = [
            'full_name' => $user->full_name,
            'otp' => $otp,
            'site_title' => setting('site_title', 'global') ?? config('app.name'),
            'expires_in' => '15 minutes',
            'site_url' => route('home'),
        ];

        $fromAddress = setting('site_email', 'global') ?? config('mail.from.address');
        $fromName = setting('site_title', 'global') ?? config('mail.from.name');

        Log::info('Attempting to send OTP email', [
            'user_id' => $user->id,
            'email' => $user->email,
            'from' => $fromAddress,
            'template' => 'emails.withdraw_otp'
        ]);

        Mail::send('emails.withdraw_otp', $data, function ($message) use ($user, $fromAddress, $fromName) {
            $message->to($user->email)
                   ->from($fromAddress, $fromName)
                   ->subject('Withdrawal OTP Verification');
        });

        // Check for failures (alternative method for Laravel 8+)
        if (Mail::failures()) {
            throw new \Exception('Failed to send to: ' . implode(', ', Mail::failures()));
        }

        Log::info('OTP email sent successfully', [
            'user_id' => $user->id,
            'email' => $user->email
        ]);

        return true;
    } catch (\Exception $e) {
        Log::error('OTP email failed', [
            'user_id' => $user->id,
            'email' => $user->email,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        return false;
    }
}
    /**
     * Resend OTP with rate limiting
     *
     * @return RedirectResponse
     */
    public function resendOtp()
    {
        $otpRecord = WithdrawOtp::where('user_id', auth()->id())
            ->where('verified', false)
            ->first();

        if (!$otpRecord) {
            notify()->error(__('No pending withdrawal request found'), 'Error');
            return redirect()->route('user.withdraw.view');
        }

        // Check if the last OTP was sent less than 1 minute ago
        if ($otpRecord->updated_at && $otpRecord->updated_at->addMinute() > now()) {
            notify()->error(__('Please wait at least 1 minute before requesting another OTP'), 'Error');
            return redirect()->route('user.withdraw.verify.otp');
        }

        // Generate new OTP
        $otp = rand(100000, 999999);
        
        // Update OTP record
        $otpRecord->update([
            'otp' => $otp,
            'expires_at' => now()->addMinutes(15),
        ]);

        // Send OTP
        $emailSent = $this->sendWithdrawOtp(auth()->user(), $otp);
        
        if ($emailSent) {
            notify()->success(__('OTP has been resent to your email address'), 'Success');
        } else {
            notify()->error(__('Failed to send OTP. Please try again later'), 'Error');
        }
        
        return redirect()->route('user.withdraw.verify.otp');
    }

    /**
     * @return Application|Factory|View
     */
    public function withdraw()
    {
        // Check global withdraw permission
        if (!setting('user_withdraw', 'permission')) {
            notify()->error(__('Withdraw is currently disabled by admin.'), 'Withdraw Disabled');
            return redirect()->back();
        }

        // Check user's withdraw status
        if (!\Auth::user()->withdraw_status) {
            notify()->error(__('Your withdraw has been disabled by admin. Please contact support for assistance.'), 'Withdraw Disabled');
            return redirect()->route('user.ticket.new');
        }

        // KYC Verification Check
        if (setting('kyc_verification', 'permission') && \Auth::user()->kyc != \App\Enums\KYCStatus::Verified->value) {
            notify()->error(__('KYC verification is required to withdraw funds. Please complete your KYC verification first.'), 'KYC Required');
            return redirect()->route('user.kyc');
        }

        $accounts = WithdrawAccount::where('user_id', \Auth::id())->get();
        $accounts = $accounts->reject(function ($value, $key) {
            return !$value->method->status;
        });

        return view('frontend.shahdeveloper.user.withdraw.now', compact('accounts'));
    }

    public function withdrawLog()
    {
        $withdraws = Transaction::search(request('query'), function ($query) {
            $query->where('user_id', auth()->user()->id)
                ->whereIn('type', ['withdraw', 'withdraw_auto'])
                ->when(request('date'), function ($query) {
                    $query->whereDay('created_at', '=', Carbon::parse(request('date'))->format('d'));
                });
        })->where('user_id', auth()->user()->id)->orderBy('created_at', 'desc')->paginate(10)->withQueryString();

        return view('frontend.shahdeveloper.user.withdraw.log', compact('withdraws'));
    }

    /**
     * Show withdraw history
     */
    public function history()
    {
        \Log::info('WithdrawController@history method called');
        \Log::info('Request URL: ' . request()->url());
        \Log::info('Request Route: ' . request()->route()->getName());
        
        $user = Auth::user();
        $currency = setting('site_currency', 'global');
        $currencySymbol = setting('currency_symbol', 'global');
        
        \Log::info('User: ' . $user->id . ', Currency: ' . $currency . ', Symbol: ' . $currencySymbol);
        
        // Get withdraw requests for this user
        $withdrawals = WithdrawRequest::where('user_id', $user->id)
            ->with(['withdrawMethod'])
            ->orderBy('created_at', 'desc')
            ->paginate(12);
        
        // Debug: Log the withdrawals count
        \Log::info('Withdrawals count: ' . $withdrawals->count());
        \Log::info('User ID: ' . $user->id);
        
        // If no withdraw transactions found, show all transactions for debugging
        if ($withdrawals->count() == 0) {
            \Log::info('No withdraw transactions found, showing all user transactions');
            $withdrawals = Transaction::where('user_id', $user->id)
                ->orderBy('created_at', 'desc')
                ->paginate(12);
        }
        
        // Ensure withdrawals is not null
        if (!$withdrawals) {
            $withdrawals = collect();
        }
        
        \Log::info('Final withdrawals count before view: ' . $withdrawals->count());
        
        return view('frontend.shahdeveloper.user.withdraw.history', compact('withdrawals', 'currency', 'currencySymbol'));
    }
}