<?php

namespace App\Models;

use App\Enums\InvestStatus;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Invest extends Model
{
    use HasFactory;

    protected $guarded = ['id'];

    protected $appends = ['created_time', 'is_cancel', 'dynamic_interest', 'progress_percentage'];

    protected $casts = [
        'status' => InvestStatus::class,
        'is_cancelled' => 'boolean',
        'cancelled_at' => 'datetime',
        'cancellation_deduction' => 'decimal:2',
        'refund_amount' => 'decimal:2',
    ];

    public function schema()
    {
        return $this->belongsTo(Schema::class, 'schema_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'user_id')->withDefault();
    }

    public function roiSchedules()
    {
        return $this->hasMany(RoiSchedule::class, 'invest_id');
    }

    public function activeRoiSchedule()
    {
        return $this->hasOne(RoiSchedule::class, 'invest_id')->where('status', 'pending');
    }

    public function readyRoiSchedule()
    {
        return $this->hasOne(RoiSchedule::class, 'invest_id')->where('status', 'ready');
    }

    /**
     * Get ROI from original investment (NOT from schema)
     * This ensures users keep their original plan ROI even if schema changes
     */
    public function getDynamicInterestAttribute()
    {
        // Always use the original investment interest, not the schema interest
        return $this->interest ?? 0;
    }

    /**
     * Get interest type from original investment (NOT from schema)
     */
    public function getDynamicInterestTypeAttribute()
    {
        // Always use the original investment interest type, not the schema interest type
        return $this->interest_type ?? 'percentage';
    }

    /**
     * Calculate profit based on dynamic ROI
     */
    public function getCalculatedProfitAttribute()
    {
        $interest = $this->dynamic_interest;
        $interestType = $this->dynamic_interest_type;
        
        if ($interestType == 'percentage') {
            return ($interest * $this->invest_amount) / 100;
        }
        
        return $interest; // Fixed amount
    }

    public function getCreatedAtAttribute($value)
    {
        return date('M d, Y H:i', strtotime($value));
    }

    public function getNextProfitTimeAttribute($value)
    {
        // Temporarily return raw value to fix filtering issues
        return $value;
        // return date('M d, Y H:i', strtotime($value));
    }

    public function getCreatedTimeAttribute(): string
    {
        return Carbon::parse($this->attributes['created_at'])->format('M d Y h:i');
    }

    public function getIsCancelAttribute(): string
    {
        if ($this->schema->schema_cancel) {
            $expiryTime = Carbon::parse($this->created_at)->addMinute($this->schema->expiry_minute)->format('M d Y h:i');
            $now = Carbon::now()->format('M d Y h:i');
            if ($expiryTime >= $now) {
                return true;
            }
            return false;
        }
        return false;
    }

    /**
     * Get progress percentage of investment
     */
    public function getProgressPercentageAttribute()
    {
        if ($this->status === InvestStatus::Completed) {
            return 100;
        }

        $startDate = Carbon::parse($this->created_at);
        $endDate = $this->end_date ? Carbon::parse($this->end_date) : $startDate->copy()->addDays($this->schema->return_period);
        $now = Carbon::now();

        if ($now >= $endDate) {
            return 100;
        }

        $totalDuration = $endDate->diffInDays($startDate);
        $elapsedDuration = $now->diffInDays($startDate);

        if ($totalDuration <= 0) {
            return 100;
        }

        return min(100, round(($elapsedDuration / $totalDuration) * 100, 2));
    }

    /**
     * Calculate ROI amount based on percentage
     */
    public function calculateRoi($percentage = null)
    {
        $roiPercentage = $percentage ?? $this->schema->return_interest;
        return ($this->invest_amount * $roiPercentage) / 100;
    }
}
