<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Invest;
use App\Models\User;
use App\Models\Schema;
use App\Models\Transaction;
use App\Models\ProfitDistribution;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProfitController extends Controller
{
    /**
     * Manual Profit Distribution Page
     */
    public function manualDistribution()
    {
        // Get all active investments
        $activeInvestments = Invest::with(['user', 'schema'])
            ->where('status', 'ongoing')
            ->orderBy('next_profit_time', 'asc')
            ->get();

        // Get all schemas for reference
        $schemas = Schema::where('status', true)->get();

        // Calculate total pending profit
        $totalPendingProfit = 0;
        foreach ($activeInvestments as $investment) {
            $calculateInterest = ($investment->interest * $investment->invest_amount) / 100;
            $interest = $investment->interest_type != 'percentage' ? $investment->interest : $calculateInterest;
            $totalPendingProfit += $interest;
        }

        return view('backend.profit.manual-distribution', compact(
            'activeInvestments',
            'schemas',
            'totalPendingProfit'
        ));
    }

    /**
     * Distribute Profit to Specific Investment
     */
    public function distributeProfit(Request $request)
    {
        $request->validate([
            'investment_id' => 'required|exists:invests,id',
            'profit_amount' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            $investment = Invest::with(['user', 'schema'])->findOrFail($request->investment_id);
            $user = $investment->user;
            $profitAmount = $request->profit_amount;

            // Update investment
            $nextProfitTime = Carbon::now()->addHour($investment->period_hours);
            
            $updateData = [
                'next_profit_time' => $nextProfitTime,
                'last_profit_time' => Carbon::now(),
                'number_of_period' => ($investment->number_of_period - 1),
                'already_return_profit' => ($investment->already_return_profit + 1),
                'total_profit_amount' => ($investment->total_profit_amount + $profitAmount),
            ];

            if ($investment->return_type == 'lifetime') {
                $investment->update($updateData);
                $user->increment('profit_balance', $profitAmount);
            } else {
                if ($investment->number_of_period > 0) {
                    if ($investment->number_of_period == 1) {
                        $updateData['status'] = 'completed';
                        
                        if ($investment->capital_back == 1) {
                            $user->increment('balance', $investment->invest_amount);
                        }
                    }
                    
                    $investment->update($updateData);
                    $user->increment('profit_balance', $profitAmount);
                }
            }

            // Create transaction record
            $transaction = new Transaction();
            $transaction->user_id = $user->id;
            $transaction->amount = $profitAmount;
            $transaction->type = 'interest';
            $transaction->status = 'success';
            $transaction->description = $investment->schema->name . ' Plan Interest (Manual)';
            $transaction->created_at = Carbon::now();
            $transaction->save();

            // Record distribution history
            $distribution = new ProfitDistribution();
            $distribution->investment_id = $investment->id;
            $distribution->user_id = $user->id;
            $distribution->profit_amount = $profitAmount;
            $distribution->distributed_by = auth()->id();
            $distribution->distribution_time = Carbon::now();
            $distribution->notes = $request->notes;
            $distribution->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Profit distributed successfully!',
                'data' => [
                    'user' => $user->username,
                    'amount' => $profitAmount,
                    'next_profit_time' => $nextProfitTime->format('Y-m-d H:i:s')
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk Profit Distribution
     */
    public function bulkDistribute(Request $request)
    {
        $request->validate([
            'investment_ids' => 'required|array',
            'investment_ids.*' => 'exists:invests,id'
        ]);

        try {
            DB::beginTransaction();

            $distributedCount = 0;
            $totalProfit = 0;
            $errors = [];

            foreach ($request->investment_ids as $investmentId) {
                try {
                    $investment = Invest::with(['user', 'schema'])->findOrFail($investmentId);
                    
                    // Check if profit is due
                    if ($investment->next_profit_time > Carbon::now()) {
                        continue;
                    }

                    $calculateInterest = ($investment->interest * $investment->invest_amount) / 100;
                    $profitAmount = $investment->interest_type != 'percentage' ? $investment->interest : $calculateInterest;

                    // Update investment
                    $nextProfitTime = Carbon::now()->addHour($investment->period_hours);
                    
                    $updateData = [
                        'next_profit_time' => $nextProfitTime,
                        'last_profit_time' => Carbon::now(),
                        'number_of_period' => ($investment->number_of_period - 1),
                        'already_return_profit' => ($investment->already_return_profit + 1),
                        'total_profit_amount' => ($investment->total_profit_amount + $profitAmount),
                    ];

                    if ($investment->return_type == 'lifetime') {
                        $investment->update($updateData);
                        $investment->user->increment('profit_balance', $profitAmount);
                    } else {
                        if ($investment->number_of_period > 0) {
                            if ($investment->number_of_period == 1) {
                                $updateData['status'] = 'completed';
                                
                                if ($investment->capital_back == 1) {
                                    $investment->user->increment('balance', $investment->invest_amount);
                                }
                            }
                            
                            $investment->update($updateData);
                            $investment->user->increment('profit_balance', $profitAmount);
                        }
                    }

                    // Create transaction
                    $transaction = new Transaction();
                    $transaction->user_id = $investment->user->id;
                    $transaction->amount = $profitAmount;
                    $transaction->type = 'interest';
                    $transaction->status = 'success';
                    $transaction->description = $investment->schema->name . ' Plan Interest (Bulk)';
                    $transaction->created_at = Carbon::now();
                    $transaction->save();

                    // Record distribution
                    $distribution = new ProfitDistribution();
                    $distribution->investment_id = $investment->id;
                    $distribution->user_id = $investment->user->id;
                    $distribution->profit_amount = $profitAmount;
                    $distribution->distributed_by = auth()->id();
                    $distribution->distribution_time = Carbon::now();
                    $distribution->notes = 'Bulk distribution';
                    $distribution->save();

                    $distributedCount++;
                    $totalProfit += $profitAmount;

                } catch (\Exception $e) {
                    $errors[] = "Investment ID {$investmentId}: " . $e->getMessage();
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Bulk distribution completed! {$distributedCount} investments processed.",
                'data' => [
                    'distributed_count' => $distributedCount,
                    'total_profit' => $totalProfit,
                    'errors' => $errors
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'success' => false,
                'message' => 'Bulk distribution failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Distribution History
     */
    public function distributionHistory()
    {
        $distributions = ProfitDistribution::with(['user', 'investment.schema'])
            ->orderBy('distribution_time', 'desc')
            ->paginate(50);

        return view('backend.profit.distribution-history', compact('distributions'));
    }

    /**
     * All Profits (existing method)
     */
    public function allProfits($id = null)
    {
        if ($id) {
            $user = User::findOrFail($id);
            $profits = Transaction::where('user_id', $id)
                ->where('type', 'interest')
                ->orderBy('created_at', 'desc')
                ->paginate(50);
        } else {
            $profits = Transaction::where('type', 'interest')
                ->with('user')
                ->orderBy('created_at', 'desc')
                ->paginate(50);
        }

        return view('backend.profit.all-profits', compact('profits', 'user'));
    }
}
