<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Invest;
use App\Models\RoiSchedule;
use Carbon\Carbon;

class CreateRoiSchedulesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        $this->command->info('Creating ROI schedules for existing investments...');
        
        // Get all ongoing investments without ROI schedules
        $investments = Invest::where('status', 'ongoing')
            ->whereDoesntHave('roiSchedules')
            ->with(['user', 'schema'])
            ->get();
            
        $this->command->info("Found {$investments->count()} investments without ROI schedules");
        
        $createdCount = 0;
        
        foreach ($investments as $investment) {
            if ($investment->schema) {
                $roiAmount = ($investment->invest_amount * $investment->schema->return_interest) / 100;
                $totalCycles = $investment->schema->number_of_period ?? 30;
                
                RoiSchedule::create([
                    'invest_id' => $investment->id,
                    'user_id' => $investment->user_id,
                    'schema_id' => $investment->schema_id,
                    'invest_amount' => $investment->invest_amount,
                    'roi_percentage' => $investment->schema->return_interest,
                    'roi_amount' => $roiAmount,
                    'roi_start_time' => Carbon::parse($investment->created_at),
                    'roi_end_time' => Carbon::parse($investment->created_at)->addHours(24),
                    'status' => 'pending',
                    'roi_cycle' => 1,
                    'total_cycles' => $totalCycles
                ]);
                
                $createdCount++;
                $this->command->line("Created ROI schedule for Investment ID: {$investment->id} - User: {$investment->user->username} - Amount: $" . number_format($roiAmount, 2));
            } else {
                $this->command->warn("Skipping Investment ID: {$investment->id} - Schema not found");
            }
        }
        
        $this->command->info("Created {$createdCount} ROI schedules for existing investments.");
    }
}
