<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cookie;

class DebugReferralRegistrationSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        echo "🔍 Debugging Referral Registration Flow...\n";
        
        // Step 1: Check referral link
        $referralLink = DB::table('referral_links')->where('user_id', 233)->first();
        if (!$referralLink) {
            echo "❌ No referral link found for user 233\n";
            return;
        }
        
        echo "🔗 Referral Link: {$referralLink->code}\n";
        echo "🔗 Referral Link ID: {$referralLink->id}\n";
        echo "🔗 Referral User ID: {$referralLink->user_id}\n";
        
        // Step 2: Simulate referral flow
        echo "\n🔍 Simulating Referral Flow:\n";
        
        // Simulate visiting /register?invite=code
        $inviteCode = $referralLink->code;
        echo "  - User visits: /register?invite={$inviteCode}\n";
        
        // Check if middleware would set cookie
        $referralExists = DB::table('referral_links')->where('code', $inviteCode)->exists();
        echo "  - Referral link exists: " . ($referralExists ? 'Yes' : 'No') . "\n";
        
        if ($referralExists) {
            $referral = DB::table('referral_links')->where('code', $inviteCode)->first();
            echo "  - Referral user ID: {$referral->user_id}\n";
            echo "  - Referral user exists: " . (DB::table('users')->where('id', $referral->user_id)->exists() ? 'Yes' : 'No') . "\n";
            
            // Check if user can be referrer
            $user = DB::table('users')->where('id', $referral->user_id)->first();
            echo "  - User ref_id: " . ($user->ref_id ?? 'NULL') . "\n";
            
            if ($user->ref_id === null) {
                echo "  ✅ User can be referrer (ref_id is NULL)\n";
            } else {
                echo "  ❌ User cannot be referrer (ref_id is {$user->ref_id})\n";
            }
        }
        
        // Step 3: Check registration process
        echo "\n🔍 Registration Process Check:\n";
        
        // Check if ref_id field is fillable
        echo "  - Checking if ref_id field exists in users table...\n";
        
        try {
            $columns = DB::select('DESCRIBE users');
            $refIdExists = false;
            foreach ($columns as $column) {
                if ($column->Field === 'ref_id') {
                    $refIdExists = true;
                    echo "  ✅ ref_id column exists in users table\n";
                    break;
                }
            }
            
            if (!$refIdExists) {
                echo "  ❌ ref_id column NOT found in users table\n";
            }
        } catch (\Exception $e) {
            echo "  ❌ Error checking table structure: " . $e->getMessage() . "\n";
        }
        
        // Step 4: Test referral creation manually
        echo "\n🧪 Manual Referral Creation Test:\n";
        
        $currentCount = DB::table('users')->where('ref_id', 233)->count();
        echo "  - Current referrals for user 233: {$currentCount}\n";
        
        // Create test user manually
        $testUsername = 'manualtest_' . time();
        try {
            $userId = DB::table('users')->insertGetId([
                'username' => $testUsername,
                'email' => $testUsername . '@test.com',
                'first_name' => 'Manual',
                'last_name' => 'Test',
                'password' => bcrypt('password123'),
                'ref_id' => 233,
                'ranking_id' => 1,
                'rankings' => json_encode([1]),
                'status' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            
            echo "  ✅ Created manual test user: {$testUsername} (ID: {$userId})\n";
            
            $newCount = DB::table('users')->where('ref_id', 233)->count();
            echo "  📊 New referral count: {$newCount}\n";
            
            if ($newCount > $currentCount) {
                echo "  🎉 SUCCESS: Manual referral creation works!\n";
                echo "  💡 Issue is in the registration flow, not database\n";
            } else {
                echo "  ❌ FAILED: Manual referral creation failed\n";
            }
            
        } catch (\Exception $e) {
            echo "  ❌ ERROR: " . $e->getMessage() . "\n";
        }
        
        echo "\n🎯 Summary:\n";
        echo "  - Referral system: " . ($refIdExists ? 'Database OK' : 'Database Issue') . "\n";
        echo "  - Manual creation: " . ($newCount > $currentCount ? 'Works' : 'Failed') . "\n";
        echo "  - Issue location: Registration flow (not database)\n";
    }
}

