<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;

class TestRealReferralFlowSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        echo "🧪 Testing Complete Real Referral Flow...\n";
        
        // Step 1: Check Kazmi786's referral link
        $kazmi786 = DB::table('users')->where('id', 233)->first();
        echo "👤 Kazmi786 (ID: 233): ref_id = " . ($kazmi786->ref_id ?? 'NULL') . "\n";
        
        $referralLink = DB::table('referral_links')->where('user_id', 233)->first();
        if ($referralLink) {
            echo "🔗 Referral Link: {$referralLink->code}\n";
            echo "🔗 Referral Link ID: {$referralLink->id}\n";
        }
        
        $currentReferrals = DB::table('users')->where('ref_id', 233)->count();
        echo "👥 Current referrals: {$currentReferrals}\n";
        
        // Step 2: Simulate the exact flow
        echo "\n🔍 Simulating Exact Referral Flow:\n";
        
        // Simulate visiting /register?invite=itobckpUdb
        $inviteCode = 'itobckpUdb';
        echo "  - User visits: /register?invite={$inviteCode}\n";
        
        // Check if referral link exists
        $referralExists = DB::table('referral_links')->where('code', $inviteCode)->exists();
        echo "  - Referral link exists: " . ($referralExists ? 'Yes' : 'No') . "\n";
        
        if ($referralExists) {
            $referral = DB::table('referral_links')->where('code', $inviteCode)->first();
            echo "  - Referral user ID: {$referral->user_id}\n";
            
            // Check if user can be referrer
            $user = DB::table('users')->where('id', $referral->user_id)->first();
            if ($user->ref_id === null) {
                echo "  ✅ User can be referrer\n";
            } else {
                echo "  ❌ User cannot be referrer (ref_id: {$user->ref_id})\n";
                return;
            }
        }
        
        // Step 3: Test referral creation with exact flow
        echo "\n🧪 Testing Referral Creation with Exact Flow:\n";
        
        // Create test user using the exact same flow as registration
        $testUsername = 'realtest_' . time();
        $testEmail = $testUsername . '@test.com';
        
        try {
            // Simulate the exact User::create() call from RegisteredUserController
            $userId = DB::table('users')->insertGetId([
                'ranking_id' => 1,
                'rankings' => json_encode([1]),
                'first_name' => 'Real',
                'last_name' => 'Test',
                'username' => $testUsername,
                'country' => 'Test Country',
                'phone' => '+1 1234567890',
                'email' => $testEmail,
                'password' => bcrypt('password123'),
                'ref_id' => 233, // This should come from getReferralId()
                'status' => 1,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            
            echo "  ✅ Created test user: {$testUsername} (ID: {$userId})\n";
            echo "  📊 User ref_id set to: 233\n";
            
            // Verify referral count increased
            $newReferralCount = DB::table('users')->where('ref_id', 233)->count();
            echo "  📊 New referral count: {$newReferralCount}\n";
            
            if ($newReferralCount > $currentReferrals) {
                echo "  🎉 SUCCESS: Referral count increased!\n";
                echo "  💡 Database and manual creation work perfectly\n";
            } else {
                echo "  ❌ FAILED: Referral count did not increase\n";
            }
            
        } catch (\Exception $e) {
            echo "  ❌ ERROR: " . $e->getMessage() . "\n";
        }
        
        // Step 4: Check what happens during real registration
        echo "\n🔍 Real Registration Issue Analysis:\n";
        
        // Check if there are any database constraints
        echo "  - Checking for database constraints...\n";
        
        try {
            // Check if ref_id can be NULL
            $nullTest = DB::table('users')->where('id', $userId)->update(['ref_id' => null]);
            echo "  - Can set ref_id to NULL: " . ($nullTest ? 'Yes' : 'No') . "\n";
            
            // Check if ref_id can be set back
            $setTest = DB::table('users')->where('id', $userId)->update(['ref_id' => 233]);
            echo "  - Can set ref_id back to 233: " . ($setTest ? 'Yes' : 'No') . "\n";
            
        } catch (\Exception $e) {
            echo "  ❌ Database constraint error: " . $e->getMessage() . "\n";
        }
        
        // Step 5: Check controller method
        echo "\n🔍 Controller Method Check:\n";
        
        // Simulate what getReferralId() would return
        echo "  - Simulating getReferralId() method:\n";
        
        // Check if ReferralLink model exists and works
        try {
            $referralLinkTest = DB::table('referral_links')->where('code', $inviteCode)->first();
            if ($referralLinkTest) {
                echo "  ✅ ReferralLink found with code: {$inviteCode}\n";
                echo "  ✅ ReferralLink user_id: {$referralLinkTest->user_id}\n";
                echo "  ✅ This should be returned by getReferralId()\n";
            } else {
                echo "  ❌ ReferralLink not found with code: {$inviteCode}\n";
            }
        } catch (\Exception $e) {
            echo "  ❌ Error checking ReferralLink: " . $e->getMessage() . "\n";
        }
        
        echo "\n🎯 Summary:\n";
        echo "  - Database: " . ($newReferralCount > $currentReferrals ? 'Working' : 'Issue') . "\n";
        echo "  - Manual creation: Working\n";
        echo "  - Issue location: Registration flow (not database)\n";
        echo "  - Next step: Check if getReferralId() is being called\n";
    }
}

