@extends('backend.layouts.app')

@section('title', 'ROI Tracking System')

@section('content')
<style>
.action-button {
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.action-button:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 12px rgba(0,0,0,0.3);
}

.action-button:active {
    transform: translateY(-1px);
}

.action-button::before {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
    transition: left 0.5s;
}

.action-button:hover::before {
    left: 100%;
}

/* Make sure buttons are clickable */
.btn {
    pointer-events: auto !important;
    cursor: pointer !important;
}

/* Debug styling */
.debug-info {
    background: #f8f9fa;
    border: 1px solid #dee2e6;
    border-radius: 5px;
    padding: 10px;
    margin: 10px 0;
    font-family: monospace;
    font-size: 12px;
}
</style>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title">ROI Tracking Dashboard</h4>
                </div>
                <div class="card-body">
                    <!-- Statistics Cards -->
                    <div class="row mb-4">
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-info">
                                <div class="inner">
                                    <h3>{{ $todayRoiData['due_today'] }}</h3>
                                    <p>ROI Due Today</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-clock"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-success">
                                <div class="inner">
                                    <h3>{{ $todayRoiData['processed_today'] }}</h3>
                                    <p>Processed Today</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-check"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-warning">
                                <div class="inner">
                                    <h3>${{ number_format($todayRoiData['total_roi_today'], 2) }}</h3>
                                    <p>Total ROI Today</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-dollar-sign"></i>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-6">
                            <div class="small-box bg-danger">
                                <div class="inner">
                                    <h3>{{ $statistics['users_without_investments'] }}</h3>
                                    <p>Users Without Investment</p>
                                </div>
                                <div class="icon">
                                    <i class="fas fa-users"></i>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="row mb-3">
                        <div class="col-12">
                            <div class="text-center">
                                <button type="button" class="btn btn-success btn-lg px-5 py-3 action-button me-3" onclick="bulkDistributeRoi()" style="font-size: 18px; font-weight: bold; cursor: pointer; box-shadow: 0 4px 8px rgba(0,0,0,0.2);" onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                                    <i class="fas fa-rocket mr-2"></i> 🚀 Distribute ROI to ALL Users
                                </button>
                                <button type="button" class="btn btn-warning btn-lg px-5 py-3 action-button me-3" onclick="bulkProcessRoi()" style="font-size: 18px; font-weight: bold; cursor: pointer; box-shadow: 0 4px 8px rgba(0,0,0,0.2);" onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                                    <i class="fas fa-play mr-2"></i> Process Today's ROI
                                </button>
                                <a href="{{ route('admin.roi-tracking.distribution-history') }}" class="btn btn-info btn-lg px-5 py-3" style="font-size: 18px; font-weight: bold;">
                                    <i class="fas fa-history mr-2"></i> Distribution History
                                </a>
                                <button type="button" class="btn btn-secondary btn-lg px-5 py-3 ms-3" onclick="testButtonClick()" style="font-size: 18px; font-weight: bold; cursor: pointer;">
                                    <i class="fas fa-bug mr-2"></i> Test Button
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Warning Alert for Testing -->
                    <div class="row mb-3">
                        <div class="col-12">
                            <div class="alert alert-warning alert-dismissible fade show" role="alert">
                                <strong>⚠️ Testing Mode:</strong> Use the "Distribute ROI to ALL Users" button to give ROI to everyone with active investments. 
                                You can reverse these distributions within 24 hours using the Distribution History page.
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        </div>
                    </div>

                    <!-- Database Table Missing Warning -->
                    @if($recentDistributions->isEmpty() && !$recentDistributions->count())
                    <div class="row mb-3">
                        <div class="col-12">
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <strong>🚨 Database Table Missing:</strong> The <code>roi_distributions</code> table is not found in your database. 
                                ROI distribution will still work, but tracking and reversal features will be limited.
                                <br><br>
                                <strong>To fix this:</strong>
                                <ol class="mb-0 mt-2">
                                    <li>Run: <code>php artisan migrate</code></li>
                                    <li>Or create the table manually using the SQL provided in the README</li>
                                </ol>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        </div>
                    </div>
                    @endif

                    <!-- Debug Info (remove in production) -->
                    @if(config('app.debug'))
                    <div class="row mb-3">
                        <div class="col-12">
                            <div class="alert alert-info">
                                <strong>Debug Info:</strong><br>
                                Total Users: {{ $users->total() }}<br>
                                Current Filter: {{ request('status', 'All Users') }}<br>
                                Current Date: {{ request('date', date('Y-m-d')) }}<br>
                                Users with investments: {{ $statistics['users_with_investments'] }}<br>
                                Users without investments: {{ $statistics['users_without_investments'] }}<br>
                                Sample User Investments: 
                                @if($users->count() > 0)
                                    @php $sampleUser = $users->first(); @endphp
                                    {{ $sampleUser->full_name }} has {{ $sampleUser->invests->count() }} investments
                                    @if($sampleUser->invests->count() > 0)
                                        (Status: {{ $sampleUser->invests->first()->status->value }})
                                    @endif
                                @endif
                            </div>
                        </div>
                    </div>
                    @endif
                    
                    <!-- JavaScript Debug Info -->
                    <div class="row mb-3">
                        <div class="col-12">
                            <div class="debug-info">
                                <strong>🔧 JavaScript Debug Info:</strong><br>
                                <div id="jsDebugInfo">Loading...</div>
                                <button type="button" class="btn btn-sm btn-outline-info mt-2" onclick="updateDebugInfo()">
                                    <i class="fas fa-sync"></i> Refresh Debug Info
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Recent ROI Distributions -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5 class="card-title">
                                        <i class="fas fa-clock"></i> Recent ROI Distributions
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="table-responsive">
                                        <table class="table table-bordered table-sm">
                                            <thead>
                                                <tr>
                                                    <th>User</th>
                                                    <th>Investment</th>
                                                    <th>ROI Amount</th>
                                                    <th>Type</th>
                                                    <th>Time</th>
                                                    <th>Status</th>
                                                    <th>Actions</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                @forelse($recentDistributions as $distribution)
                                                <tr>
                                                    <td>
                                                        <div class="d-flex align-items-center">
                                                            <img src="{{ asset($distribution->user->avatar ?? 'assets/global/images/default-avatar.png') }}" 
                                                                 class="rounded-circle mr-2" width="30" height="30">
                                                            <div>
                                                                <strong>{{ $distribution->user->full_name ?? 'Unknown User' }}</strong><br>
                                                                <small class="text-muted">{{ $distribution->user->email ?? 'N/A' }}</small>
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td>
                                                        <small>{{ $distribution->investment->schema->name ?? 'Unknown Plan' }}</small><br>
                                                        <small class="text-muted">${{ number_format($distribution->investment->invest_amount ?? 0, 2) }}</small>
                                                    </td>
                                                    <td>
                                                        <span class="text-success font-weight-bold">
                                                            ${{ number_format($distribution->roi_amount, 2) }}
                                                        </span>
                                                    </td>
                                                    <td>
                                                        @php
                                                            $typeColors = [
                                                                'manual' => 'primary',
                                                                'bulk' => 'success',
                                                                'auto' => 'info'
                                                            ];
                                                            $color = $typeColors[$distribution->distribution_type] ?? 'secondary';
                                                        @endphp
                                                        <span class="badge badge-{{ $color }}">
                                                            {{ ucfirst($distribution->distribution_type) }}
                                                        </span>
                                                    </td>
                                                    <td>
                                                        <small>{{ \Carbon\Carbon::parse($distribution->distribution_time)->format('M d, Y H:i') }}</small><br>
                                                        <small class="text-muted">{{ \Carbon\Carbon::parse($distribution->distribution_time)->diffForHumans() }}</small>
                                                    </td>
                                                    <td>
                                                        @if($distribution->is_reversed)
                                                            <span class="badge badge-danger">Reversed</span>
                                                        @else
                                                            <span class="badge badge-success">Active</span>
                                                        @endif
                                                    </td>
                                                    <td>
                                                        @if(!$distribution->is_reversed && $distribution->canBeReversed())
                                                            <button type="button" class="btn btn-sm btn-warning" 
                                                                    onclick="showReverseModal({{ $distribution->id }}, '{{ $distribution->user->full_name ?? 'Unknown User' }}', {{ $distribution->roi_amount }})">
                                                                <i class="fas fa-undo"></i> Reverse
                                                            </button>
                                                        @elseif($distribution->is_reversed)
                                                            <small class="text-muted">Already reversed</small>
                                                        @else
                                                            <small class="text-muted">{{ $distribution->time_to_reverse }}</small>
                                                        @endif
                                                    </td>
                                                </tr>
                                                @empty
                                                <tr>
                                                    <td colspan="7" class="text-center">No ROI distributions found</td>
                                                </tr>
                                                @endforelse
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Filters -->
                    <div class="row mb-3">
                        <div class="col-md-3">
                            <label>Filter by Date:</label>
                            <input type="date" class="form-control" id="dateFilter" value="{{ request('date', date('Y-m-d')) }}">
                        </div>
                        <div class="col-md-3">
                            <label>Filter by Status:</label>
                            <select class="form-control" id="statusFilter">
                                <option value="">All Users</option>
                                <option value="due_today" {{ request('status') == 'due_today' ? 'selected' : '' }}>ROI Due Today</option>
                                <option value="no_investment" {{ request('status') == 'no_investment' ? 'selected' : '' }}>Users Without Investment</option>
                                <option value="with_investment" {{ request('status') == 'with_investment' ? 'selected' : '' }}>Users With Investment</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-primary btn-block" onclick="applyFilters()">
                                <i class="fas fa-filter"></i> Apply Filters
                            </button>
                        </div>
                        <div class="col-md-3">
                            <label>&nbsp;</label>
                            <button type="button" class="btn btn-secondary btn-block" onclick="resetFilters()">
                                <i class="fas fa-undo"></i> Reset
                            </button>
                        </div>
                    </div>

                    <!-- Users Table -->
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>User</th>
                                    <th>Investment Status</th>
                                    <th>Active Investments</th>
                                    <th>Today's ROI</th>
                                    <th>Total ROI Earned</th>
                                    <th>Next ROI Time</th>
                                    <th>Progress</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($users as $user)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <img src="{{ asset($user->avatar) }}" class="rounded-circle mr-2" width="40" height="40">
                                            <div>
                                                <strong>{{ $user->full_name }}</strong><br>
                                                <small class="text-muted">{{ $user->email }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        @php
                                            $userInvests = $user->invests;
                                            $ongoingCount = $userInvests->where('status', \App\Enums\InvestStatus::Ongoing)->count();
                                            $completedCount = $userInvests->where('status', \App\Enums\InvestStatus::Completed)->count();
                                        @endphp
                                        @if($ongoingCount > 0)
                                            <span class="badge badge-success">Active ({{ $ongoingCount }})</span>
                                        @elseif($completedCount > 0)
                                            <span class="badge badge-info">Completed ({{ $completedCount }})</span>
                                        @else
                                            <span class="badge badge-warning">No Investment</span>
                                        @endif
                                    </td>
                                    <td>
                                        @php
                                            $activeInvestments = $user->invests->where('status', \App\Enums\InvestStatus::Ongoing);
                                            $totalInvested = $activeInvestments->sum('invest_amount');
                                            $totalInvestments = $user->invests->count();
                                        @endphp
                                        <strong>{{ $activeInvestments->count() }}</strong> active / {{ $totalInvestments }} total<br>
                                        <small class="text-muted">${{ number_format($totalInvested, 2) }}</small>
                                    </td>
                                    <td>
                                        @php
                                            $todayRoi = $user->invests->where('status', \App\Enums\InvestStatus::Ongoing)
                                                ->where('next_profit_time', '>=', \Carbon\Carbon::today())
                                                ->where('next_profit_time', '<', \Carbon\Carbon::tomorrow())
                                                ->sum(function($invest) {
                                                    return $invest->interest_type == 'percentage' 
                                                        ? ($invest->interest * $invest->invest_amount) / 100 
                                                        : $invest->interest;
                                                });
                                        @endphp
                                        @if($todayRoi > 0)
                                            <span class="text-success font-weight-bold">${{ number_format($todayRoi, 2) }}</span>
                                        @else
                                            <span class="text-muted">$0.00</span>
                                        @endif
                                    </td>
                                    <td>
                                        @php
                                            $totalRoi = $user->transactions->where('type', 'interest')->sum('amount');
                                        @endphp
                                        <span class="font-weight-bold">${{ number_format($totalRoi, 2) }}</span>
                                    </td>
                                    <td>
                                        @php
                                            $nextRoi = $user->invests->where('status', \App\Enums\InvestStatus::Ongoing)->min('next_profit_time');
                                        @endphp
                                        @if($nextRoi)
                                            <small>{{ \Carbon\Carbon::parse($nextRoi)->format('M d, Y H:i') }}</small><br>
                                            <small class="text-muted">{{ \Carbon\Carbon::parse($nextRoi)->diffForHumans() }}</small>
                                        @else
                                            <span class="text-muted">No upcoming ROI</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($user->invests->where('status', \App\Enums\InvestStatus::Ongoing)->count() > 0)
                                            @php
                                                $avgProgress = $user->invests->where('status', \App\Enums\InvestStatus::Ongoing)->avg(function($invest) {
                                                    if($invest->return_type == 'lifetime') return 100;
                                                    return min(100, ($invest->already_return_profit / $invest->number_of_period) * 100);
                                                });
                                            @endphp
                                            <div class="progress" style="height: 20px;">
                                                <div class="progress-bar bg-success" style="width: {{ $avgProgress }}%">
                                                    {{ number_format($avgProgress, 1) }}%
                                                </div>
                                            </div>
                                        @else
                                            <span class="text-muted">N/A</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="btn-group">
                                            <a href="{{ route('admin.roi-tracking.user-detail', $user->id) }}" 
                                               class="btn btn-sm btn-info">
                                                <i class="fas fa-eye"></i> Details
                                            </a>
                                            @if($todayRoi > 0)
                                                <button type="button" class="btn btn-sm btn-success" 
                                                        onclick="processUserRoi({{ $user->id }})">
                                                    <i class="fas fa-play"></i> Process ROI
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="8" class="text-center">No users found</td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    <div class="d-flex justify-content-center">
                        {{ $users->links() }}
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Process ROI Modal -->
<div class="modal fade" id="processRoiModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Process ROI</h5>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div id="roiDetails"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-success" onclick="confirmProcessRoi()">Process ROI</button>
            </div>
        </div>
    </div>
</div>

<!-- Reverse ROI Modal -->
<div class="modal fade" id="reverseRoiModal" tabindex="-1" aria-labelledby="reverseRoiModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="reverseRoiModalLabel">Reverse ROI Distribution</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <strong>⚠️ Warning:</strong> This action will:
                    <ul class="mb-0 mt-2">
                        <li>Deduct the ROI amount from the user's balance</li>
                        <li>Revert the investment progress</li>
                        <li>Mark the transaction as failed</li>
                        <li>This action cannot be undone!</li>
                    </ul>
                </div>
                
                <div class="mb-3">
                    <strong>User:</strong> <span id="reverseUserName"></span><br>
                    <strong>ROI Amount:</strong> <span id="reverseRoiAmount"></span><br>
                    <strong>Distribution ID:</strong> <span id="reverseDistributionId"></span>
                </div>
                
                <div class="form-group">
                    <label for="reversalReason">Reason for Reversal <span class="text-danger">*</span></label>
                    <textarea class="form-control" id="reversalReason" rows="3" placeholder="Enter the reason for reversing this ROI distribution..." required></textarea>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="confirmReverseRoi()">
                    <i class="fas fa-undo"></i> Confirm Reversal
                </button>
            </div>
        </div>
    </div>
</div>

@endsection

@push('styles')
<style>
.action-button {
    background: linear-gradient(45deg, #28a745, #20c997);
    border: none;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
    transition: all 0.3s ease;
}

.action-button:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(40, 167, 69, 0.4);
    background: linear-gradient(45deg, #20c997, #28a745);
}
</style>
@endpush

@push('scripts')
<script>
// Test function to check if buttons are working
function testButtonClick() {
    console.log('🧪 Test button clicked!');
    alert('Test button is working! This means JavaScript is functioning.');
    
    // Test SweetAlert2
    if (typeof Swal !== 'undefined') {
        Swal.fire('Test', 'SweetAlert2 is working!', 'info');
    } else {
        alert('SweetAlert2 is NOT loaded!');
    }
}

// Function to update debug info
function updateDebugInfo() {
    const debugDiv = document.getElementById('jsDebugInfo');
    if (!debugDiv) return;
    
    let info = '';
    
    // Check SweetAlert2
    if (typeof Swal !== 'undefined') {
        info += '✅ SweetAlert2: Loaded<br>';
    } else {
        info += '❌ SweetAlert2: NOT Loaded<br>';
    }
    
    // Check CSRF token
    const csrfToken = document.querySelector('meta[name="csrf-token"]');
    if (csrfToken) {
        info += '✅ CSRF Token: Found (' + csrfToken.getAttribute('content').substring(0, 10) + '...)<br>';
    } else {
        info += '❌ CSRF Token: NOT Found<br>';
    }
    
    // Check functions
    if (typeof bulkDistributeRoi !== 'undefined') {
        info += '✅ bulkDistributeRoi: Defined<br>';
    } else {
        info += '❌ bulkDistributeRoi: NOT Defined<br>';
    }
    
    if (typeof bulkProcessRoi !== 'undefined') {
        info += '✅ bulkProcessRoi: Defined<br>';
    } else {
        info += '❌ bulkProcessRoi: NOT Defined<br>';
    }
    
    // Check buttons
    const distributeBtn = document.querySelector('button[onclick="bulkDistributeRoi()"]');
    const processBtn = document.querySelector('button[onclick="bulkProcessRoi()"]');
    
    if (distributeBtn) {
        info += '✅ Distribute Button: Found<br>';
        info += '&nbsp;&nbsp;&nbsp;&nbsp;Clickable: ' + (distributeBtn.style.pointerEvents !== 'none') + '<br>';
        info += '&nbsp;&nbsp;&nbsp;&nbsp;Cursor: ' + (distributeBtn.style.cursor || 'default') + '<br>';
    } else {
        info += '❌ Distribute Button: NOT Found<br>';
    }
    
    if (processBtn) {
        info += '✅ Process Button: Found<br>';
        info += '&nbsp;&nbsp;&nbsp;&nbsp;Clickable: ' + (processBtn.style.pointerEvents !== 'none') + '<br>';
        info += '&nbsp;&nbsp;&nbsp;&nbsp;Cursor: ' + (processBtn.style.cursor || 'default') + '<br>';
    } else {
        info += '❌ Process Button: NOT Found<br>';
    }
    
    // Check for any JavaScript errors
    info += '<br><strong>Console Log:</strong><br>';
    info += '<div id="consoleLog" style="max-height: 100px; overflow-y: auto; background: #000; color: #0f0; padding: 5px; font-family: monospace; font-size: 10px;"></div>';
    
    debugDiv.innerHTML = info;
    
    // Capture console logs
    const originalLog = console.log;
    const originalError = console.error;
    const consoleDiv = document.getElementById('consoleLog');
    
    console.log = function(...args) {
        originalLog.apply(console, args);
        if (consoleDiv) {
            consoleDiv.innerHTML += args.join(' ') + '<br>';
            consoleDiv.scrollTop = consoleDiv.scrollHeight;
        }
    };
    
    console.error = function(...args) {
        originalError.apply(console, args);
        if (consoleDiv) {
            consoleDiv.innerHTML += '<span style="color: #f00;">' + args.join(' ') + '</span><br>';
            consoleDiv.scrollTop = consoleDiv.scrollHeight;
        }
    };
}

// Debug function to check if everything is loaded
function debugSetup() {
    console.log('🔍 Debug: Checking setup...');
    
    // Check if SweetAlert2 is loaded
    if (typeof Swal === 'undefined') {
        console.error('❌ SweetAlert2 not loaded!');
        return false;
    }
    console.log('✅ SweetAlert2 loaded');
    
    // Check if CSRF token exists
    const csrfToken = document.querySelector('meta[name="csrf-token"]');
    if (!csrfToken) {
        console.error('❌ CSRF token not found!');
        return false;
    }
    console.log('✅ CSRF token found:', csrfToken.getAttribute('content'));
    
    // Check if functions are defined
    if (typeof bulkDistributeRoi === 'undefined') {
        console.error('❌ bulkDistributeRoi function not defined!');
        return false;
    }
    console.log('✅ bulkDistributeRoi function defined');
    
    if (typeof bulkProcessRoi === 'undefined') {
        console.error('❌ bulkProcessRoi function not defined!');
        return false;
    }
    console.log('✅ bulkProcessRoi function defined');
    
    console.log('🎉 All checks passed!');
    return true;
}

// Run debug on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Page loaded, running debug...');
    debugSetup();
    
    // Update debug info
    setTimeout(() => {
        updateDebugInfo();
    }, 1000);
    
    // Add click event listeners to buttons as backup
    const distributeBtn = document.querySelector('button[onclick="bulkDistributeRoi()"]');
    const processBtn = document.querySelector('button[onclick="bulkProcessRoi()"]');
    
    if (distributeBtn) {
        console.log('✅ Distribute button found, adding backup listener');
        distributeBtn.addEventListener('click', function(e) {
            console.log('🖱️ Distribute button clicked via backup listener');
            bulkDistributeRoi();
        });
    }
    
    if (processBtn) {
        console.log('✅ Process button found, adding backup listener');
        processBtn.addEventListener('click', function(e) {
            console.log('🖱️ Process button clicked via backup listener');
            bulkProcessRoi();
        });
    }
});

function applyFilters() {
    const date = document.getElementById('dateFilter').value;
    const status = document.getElementById('statusFilter').value;
    
    let url = '{{ route("admin.roi-tracking.index") }}?';
    if (date) url += 'date=' + date + '&';
    if (status) url += 'status=' + status;
    
    window.location.href = url;
}

function resetFilters() {
    window.location.href = '{{ route("admin.roi-tracking.index") }}';
}

function processUserRoi(userId) {
    // Show loading
    Swal.fire({
        title: 'Processing ROI...',
        text: 'Please wait while we process the ROI for this user',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });

    // Get user's investments due today
    fetch(`/admin/roi-tracking/user/${userId}/today-roi`)
        .then(response => response.json())
        .then(data => {
            Swal.close();
            
            if (data.investments && data.investments.length > 0) {
                let html = '<div class="table-responsive"><table class="table table-sm">';
                html += '<thead><tr><th>Plan</th><th>Amount</th><th>ROI</th></tr></thead><tbody>';
                
                data.investments.forEach(invest => {
                    html += `<tr>
                        <td>${invest.schema.name}</td>
                        <td>$${parseFloat(invest.invest_amount).toFixed(2)}</td>
                        <td>$${parseFloat(invest.roi_amount).toFixed(2)}</td>
                    </tr>`;
                });
                
                html += '</tbody></table></div>';
                html += `<p class="text-success"><strong>Total ROI: $${parseFloat(data.total_roi).toFixed(2)}</strong></p>`;
                
                document.getElementById('roiDetails').innerHTML = html;
                $('#processRoiModal').modal('show');
                
                // Store user ID for processing
                window.currentUserId = userId;
            } else {
                Swal.fire('No ROI Due', 'This user has no ROI due today.', 'info');
            }
        })
        .catch(error => {
            Swal.close();
            Swal.fire('Error', 'Failed to load ROI details.', 'error');
        });
}

function confirmProcessRoi() {
    const userId = window.currentUserId;
    
    Swal.fire({
        title: 'Confirm ROI Processing',
        text: 'Are you sure you want to process ROI for this user?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'Yes, Process',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            processRoi(userId);
        }
    });
}

function processRoi(userId) {
    Swal.fire({
        title: 'Processing ROI...',
        text: 'Please wait',
        allowOutsideClick: false,
        didOpen: () => {
            Swal.showLoading();
        }
    });

    fetch('/admin/roi-tracking/process-roi', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
            user_id: userId
        })
    })
    .then(response => response.json())
    .then(data => {
        Swal.close();
        
        if (data.success) {
            Swal.fire('Success', data.message, 'success').then(() => {
                location.reload();
            });
        } else {
            Swal.fire('Error', data.message || 'Failed to process ROI', 'error');
        }
    })
    .catch(error => {
        Swal.close();
        Swal.fire('Error', 'Failed to process ROI', 'error');
    });
}

function bulkProcessRoi() {
    Swal.fire({
        title: 'Bulk Process ROI',
        text: 'This will process ROI for all investments due today. Continue?',
        icon: 'question',
        showCancelButton: true,
        confirmButtonText: 'Yes, Process All',
        cancelButtonText: 'Cancel'
    }).then((result) => {
        if (result.isConfirmed) {
            Swal.fire({
                title: 'Processing...',
                text: 'Please wait while we process all ROI',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            fetch('/admin/roi-tracking/bulk-process-roi', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                Swal.close();
                
                if (data.success) {
                    Swal.fire('Success', data.message, 'success').then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire('Error', data.message || 'Failed to process ROI', 'error');
                }
            })
            .catch(error => {
                Swal.close();
                Swal.fire('Error', 'Failed to process ROI', 'error');
            });
        }
    });
}

// New function for bulk ROI distribution to ALL users
function bulkDistributeRoi() {
    Swal.fire({
        title: '🚀 Distribute ROI to ALL Users',
        html: `
            <div class="text-left">
                <p><strong>This action will:</strong></p>
                <ul class="text-left">
                    <li>Give ROI to <strong>ALL users</strong> with active investments</li>
                    <li>Update their balances immediately</li>
                    <li>Process all ongoing investments</li>
                    <li>Create transaction records for each user</li>
                </ul>
                <div class="alert alert-warning mt-3">
                    <strong>⚠️ Testing Mode:</strong> You can reverse these distributions within 24 hours using the Distribution History page.
                </div>
            </div>
        `,
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: '🚀 Yes, Distribute to ALL Users!',
        cancelButtonText: 'Cancel',
        confirmButtonColor: '#28a745',
        cancelButtonColor: '#6c757d'
    }).then((result) => {
        if (result.isConfirmed) {
            Swal.fire({
                title: '🚀 Distributing ROI...',
                html: `
                    <div class="text-center">
                        <div class="spinner-border text-primary mb-3" role="status">
                            <span class="visually-hidden">Loading...</span>
                        </div>
                        <p>Processing ROI for all users with active investments...</p>
                        <p class="text-muted">This may take a few moments</p>
                    </div>
                `,
                allowOutsideClick: false,
                showConfirmButton: false
            });

            fetch('/admin/roi-tracking/bulk-distribute-roi', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                Swal.close();
                
                if (data.success) {
                    Swal.fire({
                        title: '🎉 Success!',
                        html: `
                            <div class="text-center">
                                <div class="text-success mb-3">
                                    <i class="fas fa-check-circle fa-3x"></i>
                                </div>
                                <h5>ROI Distributed Successfully!</h5>
                                <p><strong>${data.data.distributed_count}</strong> investments processed</p>
                                <p><strong>Total ROI Amount:</strong> $${parseFloat(data.data.total_roi_amount).toFixed(2)}</p>
                                ${data.data.errors.length > 0 ? `<div class="alert alert-warning mt-3"><strong>Warnings:</strong> ${data.data.errors.length} errors occurred</div>` : ''}
                            </div>
                        `,
                        icon: 'success',
                        confirmButtonText: 'View Results',
                        showCancelButton: true,
                        cancelButtonText: 'Close'
                    }).then((result) => {
                        if (result.isConfirmed) {
                            location.reload();
                        } else {
                            location.reload();
                        }
                    });
                } else {
                    Swal.fire('❌ Error', data.message || 'Failed to distribute ROI', 'error');
                }
            })
            .catch(error => {
                Swal.close();
                Swal.fire('❌ Error', 'Failed to distribute ROI: ' + error.message, 'error');
            });
        }
    });
}

// Function to show reverse ROI modal
function showReverseModal(distributionId, userName, roiAmount) {
    document.getElementById('reverseDistributionId').textContent = distributionId;
    document.getElementById('reverseUserName').textContent = userName;
    document.getElementById('reverseRoiAmount').textContent = '$' + parseFloat(roiAmount).toFixed(2);
    document.getElementById('reversalReason').value = '';
    
    $('#reverseRoiModal').modal('show');
}

// Function to confirm ROI reversal
function confirmReverseRoi() {
    const distributionId = document.getElementById('reverseDistributionId').textContent;
    const reversalReason = document.getElementById('reversalReason').value.trim();
    
    if (!reversalReason) {
        Swal.fire('Error', 'Please provide a reason for the reversal', 'error');
        return;
    }
    
    Swal.fire({
        title: '⚠️ Confirm Reversal',
        text: 'Are you absolutely sure you want to reverse this ROI distribution? This action cannot be undone!',
        icon: 'warning',
        showCancelButton: true,
        confirmButtonText: 'Yes, Reverse It!',
        cancelButtonText: 'Cancel',
        confirmButtonColor: '#dc3545',
        cancelButtonColor: '#6c757d'
    }).then((result) => {
        if (result.isConfirmed) {
            Swal.fire({
                title: 'Reversing ROI...',
                text: 'Please wait while we reverse the distribution',
                allowOutsideClick: false,
                didOpen: () => {
                    Swal.showLoading();
                }
            });

            fetch('/admin/roi-tracking/reverse-roi', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    distribution_id: distributionId,
                    reversal_reason: reversalReason
                })
            })
            .then(response => response.json())
            .then(data => {
                Swal.close();
                
                if (data.success) {
                    Swal.fire({
                        title: '✅ Reversed Successfully!',
                        text: data.message,
                        icon: 'success',
                        confirmButtonText: 'OK'
                    }).then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire('❌ Error', data.message || 'Failed to reverse ROI', 'error');
                }
            })
            .catch(error => {
                Swal.close();
                Swal.fire('❌ Error', 'Failed to reverse ROI: ' + error.message, 'error');
            });
        }
    });
}
</script>
@endpush
