@extends('frontend::layouts.user')
@section('title')
    {{ __('Withdraw Funds') }}
@endsection

@section('content')
<div class="withdraw-container">
    <!-- Hero Section -->
    <div class="withdraw-hero">
        <div class="hero-content">
            <div class="hero-icon">💎</div>
            <h1 class="hero-title">{{ __('Quick Withdraw') }}</h1>
            <p class="hero-subtitle">{{ __('Fast, secure, and reliable withdrawals to your preferred payment method') }}</p>
        </div>
    </div>

    <!-- Main Content -->
    <div class="withdraw-main">
        <div class="withdraw-grid">
            <!-- Left Column - Form -->
            <div class="withdraw-form-section">
                <div class="glass-card">
                    <div class="card-header">
                        <h3 class="card-title">🎯 {{ __('Select Withdrawal Method') }}</h3>
                        <p class="card-subtitle">{{ __('Choose your preferred withdrawal category') }}</p>
                    </div>
                    
                    <form action="{{ route('user.withdraw.simple.store') }}" method="post" id="withdrawForm">
                        @csrf
                        
                        <!-- Method Type Selection -->
                        <div class="method-selection">
                            <div class="method-grid">
                                <div class="method-option" onclick="selectMethodType('crypto')">
                                    <input type="radio" name="method_type" value="crypto" id="crypto" style="display: none;">
                                    <div class="method-card">
                                        <div class="method-icon">🚀</div>
                                        <div class="method-content">
                                            <h4 class="method-title">{{ __('Cryptocurrency') }}</h4>
                                            <p class="method-desc">{{ __('Solana, Arbitrum, BNB, USDT') }}</p>
                                        </div>
                                        <div class="method-check">
                                            <div class="check-circle"></div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="method-option" onclick="selectMethodType('banking')">
                                    <input type="radio" name="method_type" value="banking" id="banking" style="display: none;">
                                    <div class="method-card">
                                        <div class="method-icon">🏦</div>
                                        <div class="method-content">
                                            <h4 class="method-title">{{ __('Banking Services') }}</h4>
                                            <p class="method-desc">{{ __('EasyPaisa, JazzCash, Bank Transfer') }}</p>
                                        </div>
                                        <div class="method-check">
                                            <div class="check-circle"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Crypto Methods -->
                        <div id="crypto-methods" class="methods-display" style="display: none;">
                            <div class="section-header">
                                <h4 class="section-title">🚀 {{ __('Cryptocurrency Methods') }}</h4>
                                <p class="section-subtitle">{{ __('Select your preferred crypto network') }}</p>
                            </div>
                            <div class="methods-grid">
                                @foreach($methods['crypto'] as $key => $method)
                                <div class="method-item" onclick="selectMethod('{{ $key }}', 'crypto')">
                                    <input type="radio" name="method" value="{{ $key }}" id="method_{{ $key }}" style="display: none;">
                                    <div class="method-item-card">
                                        <div class="method-item-icon">{{ $method['icon'] }}</div>
                                        <div class="method-item-content">
                                            <h5 class="method-item-title">{{ $method['name'] }}</h5>
                                            <div class="method-item-details">
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Min') }}:</span>
                                                    <span class="detail-value">{{ $currencySymbol }}{{ $method['min_amount'] }}</span>
                                                </span>
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Max') }}:</span>
                                                    <span class="detail-value">{{ $currencySymbol }}{{ $method['max_amount'] }}</span>
                                                </span>
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Fee') }}:</span>
                                                    <span class="detail-value fee-badge">{{ $method['fee_percentage'] == 0 ? 'FREE' : $method['fee_percentage'].'%' }}</span>
                                                </span>
                                            </div>
                                        </div>
                                        <div class="method-item-check">
                                            <div class="item-check-circle"></div>
                                        </div>
                                    </div>
                                </div>
                                @endforeach
                            </div>
                        </div>

                        <!-- Banking Methods -->
                        <div id="banking-methods" class="methods-display" style="display: none;">
                            <div class="section-header">
                                <h4 class="section-title">🏦 {{ __('Banking Services') }}</h4>
                                <p class="section-subtitle">{{ __('Select your preferred banking method') }}</p>
                            </div>
                            <div class="methods-grid">
                                @foreach($methods['banking'] as $key => $method)
                                <div class="method-item" onclick="selectMethod('{{ $key }}', 'banking')">
                                    <input type="radio" name="method" value="{{ $key }}" id="method_{{ $key }}" style="display: none;">
                                    <div class="method-item-card">
                                        <div class="method-item-icon">{{ $method['icon'] }}</div>
                                        <div class="method-item-content">
                                            <h5 class="method-item-title">{{ $method['name'] }}</h5>
                                            <div class="method-item-details">
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Min') }}:</span>
                                                    <span class="detail-value">{{ $currencySymbol }}{{ $method['min_amount'] }}</span>
                                                </span>
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Max') }}:</span>
                                                    <span class="detail-value">{{ $currencySymbol }}{{ $method['max_amount'] }}</span>
                                                </span>
                                                <span class="detail-item">
                                                    <span class="detail-label">{{ __('Fee') }}:</span>
                                                    <span class="detail-value fee-badge">{{ $method['fee_percentage'] == 0 ? 'FREE' : $method['fee_percentage'].'%' }}</span>
                                                </span>
                                            </div>
                                        </div>
                                        <div class="method-item-check">
                                            <div class="item-check-circle"></div>
                                        </div>
                                    </div>
                                </div>
                                @endforeach
                            </div>
                        </div>

                        <!-- Amount Input -->
                        <div id="amount-section" class="amount-section" style="display: none;">
                            <div class="section-header">
                                <h4 class="section-title">💰 {{ __('Withdrawal Amount') }}</h4>
                                <p class="section-subtitle">{{ __('Enter the amount you want to withdraw') }}</p>
                            </div>
                            
                            <div class="amount-input-group">
                                <div class="amount-input-wrapper">
                                    <label class="input-label">{{ __('Amount') }}</label>
                                    <div class="input-group">
                                        <span class="currency-symbol">{{ $currencySymbol }}</span>
                                        <input type="number" name="amount" id="amount" class="amount-input" placeholder="0.00" step="0.01" min="1">
                                    </div>
                                    <div id="amount-info" class="input-info"></div>
                                </div>
                                
                                <div class="balance-display">
                                    <div class="balance-card">
                                        <div class="balance-icon">💳</div>
                                        <div class="balance-content">
                                            <h5 class="balance-title">{{ __('Available Balance') }}</h5>
                                            <p class="balance-amount">{{ $currencySymbol }}{{ number_format($user->balance, 2) }}</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Dynamic Fields -->
                        <div id="dynamic-fields" class="dynamic-fields" style="display: none;">
                            <!-- Crypto Fields -->
                            <div id="crypto-fields" class="field-group">
                                <div class="section-header">
                                    <h4 class="section-title">🔗 {{ __('Wallet Details') }}</h4>
                                    <p class="section-subtitle">{{ __('Enter your wallet information') }}</p>
                                </div>
                                
                                <div class="fields-grid">
                                    <div class="field-item">
                                        <label class="field-label">{{ __('Wallet Address') }}</label>
                                        <input type="text" name="wallet_address" class="field-input" placeholder="{{ __('Enter your wallet address') }}">
                                    </div>
                                    <div class="field-item" id="network-field" style="display: none;">
                                        <label class="field-label">{{ __('Network') }}</label>
                                        <select name="network" class="field-select">
                                            <option value="">{{ __('Select Network') }}</option>
                                            <option value="TRC20">TRC20 (Tron)</option>
                                            <option value="ERC20">ERC20 (Ethereum)</option>
                                            <option value="BEP20">BEP20 (BSC)</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Banking Fields -->
                            <div id="banking-fields" class="field-group">
                                <div class="section-header">
                                    <h4 class="section-title">🏦 {{ __('Account Details') }}</h4>
                                    <p class="section-subtitle">{{ __('Enter your banking information') }}</p>
                                </div>
                                
                                <div class="fields-grid">
                                    <div class="field-item">
                                        <label class="field-label">{{ __('Account Number') }}</label>
                                        <input type="text" name="account_number" class="field-input" placeholder="{{ __('Enter account number') }}">
                                    </div>
                                    <div class="field-item">
                                        <label class="field-label">{{ __('Account Title') }}</label>
                                        <input type="text" name="account_title" class="field-input" placeholder="{{ __('Enter account holder name') }}">
                                    </div>
                                    <div class="field-item" id="iban-field" style="display: none;">
                                        <label class="field-label">{{ __('IBAN Number') }}</label>
                                        <input type="text" name="iban" class="field-input" placeholder="{{ __('Enter IBAN number') }}">
                                    </div>
                                    <div class="field-item" id="bank-name-field" style="display: none;">
                                        <label class="field-label">{{ __('Bank Name') }}</label>
                                        <input type="text" name="bank_name" class="field-input" placeholder="{{ __('Enter bank name') }}">
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Fee Calculation -->
                        <div id="fee-calculation" class="fee-calculation" style="display: none;">
                            <div class="fee-card">
                                <div class="fee-header">
                                    <h4 class="fee-title">📊 {{ __('Withdrawal Summary') }}</h4>
                                </div>
                                <div class="fee-details">
                                    <div class="fee-row">
                                        <span class="fee-label">{{ __('Amount') }}</span>
                                        <span class="fee-value" id="calc-amount">{{ $currencySymbol }}0.00</span>
                                    </div>
                                    <div class="fee-row">
                                        <span class="fee-label">{{ __('Fee') }}</span>
                                        <span class="fee-value" id="calc-fee">{{ $currencySymbol }}0.00</span>
                                    </div>
                                    <div class="fee-divider"></div>
                                    <div class="fee-row total">
                                        <span class="fee-label">{{ __('Total Deduction') }}</span>
                                        <span class="fee-value" id="calc-total">{{ $currencySymbol }}0.00</span>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Submit Button -->
                        <div class="submit-section">
                            <button type="submit" class="submit-btn" id="submitBtn" disabled>
                                <span class="btn-icon">🚀</span>
                                <span class="btn-text">{{ __('Submit Withdrawal Request') }}</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Right Column - Info -->
            <div class="withdraw-info-section">
                <div class="glass-card info-card">
                    <div class="info-header">
                        <h3 class="info-title">ℹ️ {{ __('Important Information') }}</h3>
                    </div>
                    
                    <div class="info-content">
                        <div class="info-item">
                            <div class="info-icon">⏰</div>
                            <div class="info-text">
                                <h5 class="info-item-title">{{ __('Processing Time') }}</h5>
                                <p class="info-item-desc">{{ __('All withdrawal requests are processed within 24 hours during business days.') }}</p>
                            </div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-icon">🔒</div>
                            <div class="info-text">
                                <h5 class="info-item-title">{{ __('Security Note') }}</h5>
                                <p class="info-item-desc">{{ __('Please ensure your account details are correct. Incorrect details may cause delays or failed transactions.') }}</p>
                            </div>
                        </div>
                        
                        <div class="info-item">
                            <div class="info-icon">💡</div>
                            <div class="info-text">
                                <h5 class="info-item-title">{{ __('Tips') }}</h5>
                                <ul class="info-list">
                                    <li>{{ __('Double-check wallet addresses for crypto withdrawals') }}</li>
                                    <li>{{ __('Ensure sufficient balance for fees') }}</li>
                                    <li>{{ __('Contact support if you need assistance') }}</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* Withdraw Page Styles */
.withdraw-container {
    min-height: 100vh;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    padding: 20px;
}

.withdraw-hero {
    text-align: center;
    margin-bottom: 40px;
    padding: 40px 20px;
}

.hero-content {
    max-width: 600px;
    margin: 0 auto;
}

.hero-icon {
    font-size: 4rem;
    margin-bottom: 20px;
    animation: float 3s ease-in-out infinite;
}

.hero-title {
    font-size: 3rem;
    font-weight: 700;
    color: white;
    margin-bottom: 15px;
    text-shadow: 0 4px 8px rgba(0,0,0,0.3);
}

.hero-subtitle {
    font-size: 1.2rem;
    color: rgba(255,255,255,0.9);
    line-height: 1.6;
}

.withdraw-main {
    max-width: 1400px;
    margin: 0 auto;
}

.withdraw-grid {
    display: grid;
    grid-template-columns: 1fr 350px;
    gap: 30px;
    align-items: start;
}

.glass-card {
    background: rgba(255,255,255,0.1);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(255,255,255,0.2);
    padding: 30px;
    box-shadow: 0 20px 40px rgba(0,0,0,0.1);
}

.card-header {
    text-align: center;
    margin-bottom: 30px;
}

.card-title {
    font-size: 1.8rem;
    font-weight: 600;
    color: white;
    margin-bottom: 10px;
}

.card-subtitle {
    color: rgba(255,255,255,0.8);
    font-size: 1rem;
}

/* Method Selection */
.method-selection {
    margin-bottom: 30px;
}

.method-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

.method-option {
    cursor: pointer;
}

.method-card {
    background: rgba(255,255,255,0.1);
    border: 2px solid transparent;
    border-radius: 15px;
    padding: 25px;
    text-align: center;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.method-card:hover {
    transform: translateY(-5px);
    border-color: rgba(255,255,255,0.3);
    background: rgba(255,255,255,0.15);
}

.method-card.selected {
    border-color: #667eea;
    background: rgba(102, 126, 234, 0.2);
}

.method-icon {
    font-size: 2.5rem;
    margin-bottom: 15px;
}

.method-content h4 {
    color: white;
    font-size: 1.3rem;
    font-weight: 600;
    margin-bottom: 8px;
}

.method-content p {
    color: rgba(255,255,255,0.8);
    font-size: 0.9rem;
    margin: 0;
}

.method-check {
    position: absolute;
    top: 15px;
    right: 15px;
}

.check-circle {
    width: 20px;
    height: 20px;
    border: 2px solid rgba(255,255,255,0.3);
    border-radius: 50%;
    transition: all 0.3s ease;
}

.method-card.selected .check-circle {
    background: #667eea;
    border-color: #667eea;
}

/* Methods Display */
.methods-display {
    margin-bottom: 30px;
}

.section-header {
    text-align: center;
    margin-bottom: 25px;
}

.section-title {
    color: white;
    font-size: 1.5rem;
    font-weight: 600;
    margin-bottom: 8px;
}

.section-subtitle {
    color: rgba(255,255,255,0.8);
    font-size: 0.95rem;
}

.methods-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
}

.method-item {
    cursor: pointer;
}

.method-item-card {
    background: rgba(255,255,255,0.1);
    border: 2px solid transparent;
    border-radius: 15px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.method-item-card:hover {
    transform: translateY(-3px);
    border-color: rgba(255,255,255,0.3);
    background: rgba(255,255,255,0.15);
}

.method-item-card.selected {
    border-color: #667eea;
    background: rgba(102, 126, 234, 0.2);
}

.method-item-icon {
    font-size: 2rem;
    flex-shrink: 0;
}

.method-item-content {
    flex: 1;
}

.method-item-title {
    color: white;
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 8px;
}

.method-item-details {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.detail-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.detail-label {
    color: rgba(255,255,255,0.7);
    font-size: 0.85rem;
}

.detail-value {
    color: white;
    font-size: 0.85rem;
    font-weight: 500;
}

.fee-badge {
    background: rgba(76, 175, 80, 0.3);
    color: #4caf50;
    padding: 2px 8px;
    border-radius: 10px;
    font-size: 0.75rem;
    font-weight: 600;
}

.method-item-check {
    flex-shrink: 0;
}

.item-check-circle {
    width: 18px;
    height: 18px;
    border: 2px solid rgba(255,255,255,0.3);
    border-radius: 50%;
    transition: all 0.3s ease;
}

.method-item-card.selected .item-check-circle {
    background: #667eea;
    border-color: #667eea;
}

/* Amount Section */
.amount-section {
    margin-bottom: 30px;
}

.amount-input-group {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

.amount-input-wrapper {
    display: flex;
    flex-direction: column;
}

.input-label {
    color: white;
    font-weight: 500;
    margin-bottom: 8px;
}

.input-group {
    position: relative;
    display: flex;
    align-items: center;
}

.currency-symbol {
    position: absolute;
    left: 15px;
    color: rgba(255,255,255,0.7);
    font-weight: 500;
    z-index: 2;
}

.amount-input {
    width: 100%;
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    border-radius: 10px;
    padding: 15px 15px 15px 40px;
    color: white;
    font-size: 1.1rem;
    font-weight: 500;
    transition: all 0.3s ease;
}

.amount-input:focus {
    outline: none;
    border-color: #667eea;
    background: rgba(255,255,255,0.15);
}

.amount-input::placeholder {
    color: rgba(255,255,255,0.5);
}

.input-info {
    margin-top: 8px;
    font-size: 0.85rem;
    color: rgba(255,255,255,0.7);
}

.balance-display {
    display: flex;
    align-items: center;
}

.balance-card {
    background: rgba(255,255,255,0.1);
    border-radius: 15px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    width: 100%;
}

.balance-icon {
    font-size: 1.5rem;
}

.balance-content h5 {
    color: rgba(255,255,255,0.8);
    font-size: 0.9rem;
    font-weight: 500;
    margin-bottom: 5px;
}

.balance-amount {
    color: white;
    font-size: 1.2rem;
    font-weight: 600;
    margin: 0;
}

/* Dynamic Fields */
.dynamic-fields {
    margin-bottom: 30px;
}

.field-group {
    margin-bottom: 25px;
}

.fields-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

.field-item {
    display: flex;
    flex-direction: column;
}

.field-label {
    color: white;
    font-weight: 500;
    margin-bottom: 8px;
}

.field-input, .field-select {
    background: rgba(255,255,255,0.1);
    border: 1px solid rgba(255,255,255,0.2);
    border-radius: 10px;
    padding: 15px;
    color: white;
    font-size: 1rem;
    transition: all 0.3s ease;
}

.field-input:focus, .field-select:focus {
    outline: none;
    border-color: #667eea;
    background: rgba(255,255,255,0.15);
}

.field-input::placeholder {
    color: rgba(255,255,255,0.5);
}

.field-select option {
    background: #2c3e50;
    color: white;
}

/* Fee Calculation */
.fee-calculation {
    margin-bottom: 30px;
}

.fee-card {
    background: rgba(255,255,255,0.1);
    border-radius: 15px;
    padding: 25px;
    border: 1px solid rgba(255,255,255,0.2);
}

.fee-header {
    text-align: center;
    margin-bottom: 20px;
}

.fee-title {
    color: white;
    font-size: 1.3rem;
    font-weight: 600;
    margin: 0;
}

.fee-details {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.fee-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
}

.fee-row.total {
    border-top: 1px solid rgba(255,255,255,0.2);
    padding-top: 15px;
    margin-top: 5px;
    font-weight: 600;
}

.fee-label {
    color: rgba(255,255,255,0.8);
    font-size: 0.95rem;
}

.fee-value {
    color: white;
    font-size: 0.95rem;
    font-weight: 500;
}

.fee-row.total .fee-label,
.fee-row.total .fee-value {
    font-size: 1.1rem;
    font-weight: 600;
}

/* Submit Section */
.submit-section {
    text-align: center;
}

.submit-btn {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    border-radius: 15px;
    padding: 18px 40px;
    font-size: 1.1rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    display: inline-flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 8px 25px rgba(102, 126, 234, 0.3);
}

.submit-btn:hover:not(:disabled) {
    transform: translateY(-3px);
    box-shadow: 0 12px 35px rgba(102, 126, 234, 0.4);
}

.submit-btn:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-icon {
    font-size: 1.2rem;
}

/* Info Section */
.info-card {
    position: sticky;
    top: 20px;
}

.info-header {
    text-align: center;
    margin-bottom: 25px;
}

.info-title {
    color: white;
    font-size: 1.5rem;
    font-weight: 600;
    margin: 0;
}

.info-content {
    display: flex;
    flex-direction: column;
    gap: 25px;
}

.info-item {
    display: flex;
    gap: 15px;
    align-items: flex-start;
}

.info-icon {
    font-size: 1.5rem;
    flex-shrink: 0;
    margin-top: 2px;
}

.info-text h5 {
    color: white;
    font-size: 1.1rem;
    font-weight: 600;
    margin-bottom: 8px;
}

.info-text p, .info-text li {
    color: rgba(255,255,255,0.8);
    font-size: 0.9rem;
    line-height: 1.5;
    margin: 0;
}

.info-list {
    margin: 0;
    padding-left: 20px;
}

.info-list li {
    margin-bottom: 5px;
}

/* Animations */
@keyframes float {
    0%, 100% { transform: translateY(0px); }
    50% { transform: translateY(-10px); }
}

/* Responsive Design */
@media (max-width: 1200px) {
    .withdraw-grid {
        grid-template-columns: 1fr;
        gap: 20px;
    }
    
    .info-card {
        position: static;
    }
}

@media (max-width: 768px) {
    .withdraw-container {
        padding: 15px;
    }
    
    .hero-title {
        font-size: 2rem;
    }
    
    .hero-subtitle {
        font-size: 1rem;
    }
    
    .method-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .methods-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .amount-input-group {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .fields-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .glass-card {
        padding: 20px;
    }
    
    .method-card, .method-item-card {
        padding: 20px;
    }
}

@media (max-width: 480px) {
    .hero-icon {
        font-size: 3rem;
    }
    
    .hero-title {
        font-size: 1.8rem;
    }
    
    .card-title {
        font-size: 1.5rem;
    }
    
    .section-title {
        font-size: 1.3rem;
    }
    
    .submit-btn {
        padding: 15px 30px;
        font-size: 1rem;
    }
}
</style>

@endsection

@section('script')
<script>
const methods = @json($methods);
let selectedMethodType = null;
let selectedMethod = null;
let selectedMethodInfo = null;

function selectMethodType(type) {
    selectedMethodType = type;
    
    // Update UI
    document.querySelectorAll('.method-card').forEach(card => {
        card.classList.remove('selected');
    });
    event.currentTarget.querySelector('.method-card').classList.add('selected');
    
    // Show/hide method sections
    document.getElementById('crypto-methods').style.display = type === 'crypto' ? 'block' : 'none';
    document.getElementById('banking-methods').style.display = type === 'banking' ? 'block' : 'none';
    
    // Clear previous method selection
    selectedMethod = null;
    selectedMethodInfo = null;
    document.querySelectorAll('input[name="method"]').forEach(input => input.checked = false);
    document.querySelectorAll('.method-item-card').forEach(card => {
        card.classList.remove('selected');
    });
    document.getElementById('amount-section').style.display = 'none';
    document.getElementById('dynamic-fields').style.display = 'none';
    document.getElementById('fee-calculation').style.display = 'none';
    document.getElementById('submitBtn').disabled = true;
}

function selectMethod(method, type) {
    selectedMethod = method;
    selectedMethodInfo = methods[type][method];
    
    // Update UI
    document.querySelectorAll('.method-item-card').forEach(card => {
        card.classList.remove('selected');
    });
    event.currentTarget.querySelector('.method-item-card').classList.add('selected');
    
    // Show amount section
    document.getElementById('amount-section').style.display = 'block';
    document.getElementById('amount-info').innerHTML = 
        `Min: {{ $currencySymbol }}${selectedMethodInfo.min_amount} | Max: {{ $currencySymbol }}${selectedMethodInfo.max_amount} | Fee: ${selectedMethodInfo.fee_percentage}%`;
    
    // Show appropriate fields
    showDynamicFields(type, method);
    
    // Update form values
    document.getElementById(type).checked = true;
    document.getElementById('method_' + method).checked = true;
}

function showDynamicFields(type, method) {
    document.getElementById('dynamic-fields').style.display = 'block';
    
    // Hide all field groups first
    document.querySelectorAll('.field-group').forEach(group => {
        group.style.display = 'none';
    });
    
    if (type === 'crypto') {
        document.getElementById('crypto-fields').style.display = 'block';
        // Show network field only for USDT
        document.getElementById('network-field').style.display = method === 'usdt' ? 'block' : 'none';
    } else if (type === 'banking') {
        document.getElementById('banking-fields').style.display = 'block';
        // Show IBAN and Bank Name only for bank transfer
        document.getElementById('iban-field').style.display = method === 'bank_transfer' ? 'block' : 'none';
        document.getElementById('bank-name-field').style.display = method === 'bank_transfer' ? 'block' : 'none';
    }
}

// Amount change handler
document.getElementById('amount').addEventListener('input', function() {
    const amount = parseFloat(this.value) || 0;
    
    if (selectedMethodInfo) {
        const fee = (amount * selectedMethodInfo.fee_percentage) / 100;
        const total = amount + fee;
        
        document.getElementById('calc-amount').textContent = `{{ $currencySymbol }}${amount.toFixed(2)}`;
        document.getElementById('calc-fee').textContent = `{{ $currencySymbol }}${fee.toFixed(2)}`;
        document.getElementById('calc-total').textContent = `{{ $currencySymbol }}${total.toFixed(2)}`;
        
        if (amount >= selectedMethodInfo.min_amount && amount <= selectedMethodInfo.max_amount) {
            document.getElementById('fee-calculation').style.display = 'block';
            document.getElementById('submitBtn').disabled = false;
        } else {
            document.getElementById('fee-calculation').style.display = 'none';
            document.getElementById('submitBtn').disabled = true;
        }
    }
});

// Form validation
document.getElementById('withdrawForm').addEventListener('submit', function(e) {
    if (!selectedMethodType || !selectedMethod) {
        e.preventDefault();
        alert('Please select a withdrawal method');
        return false;
    }
    
    const amount = parseFloat(document.getElementById('amount').value) || 0;
    if (!selectedMethodInfo || amount < selectedMethodInfo.min_amount || amount > selectedMethodInfo.max_amount) {
        e.preventDefault();
        alert(`Amount must be between {{ $currencySymbol }}${selectedMethodInfo.min_amount} and {{ $currencySymbol }}${selectedMethodInfo.max_amount}`);
        return false;
    }
});
</script>
@endsection
