@extends('frontend.shahdeveloper.layouts.user')
@section('title')
{{ __('Referral Tree') }}
@endsection
@section('subtitle')
Track your referral network and earnings
@endsection
@section('content')
@php
    $currency = setting('currency_symbol', 'global', '$');
@endphp

<!-- InvestFlow Referral Tree Page -->
<div class="page-content">
<div class="referral-tree-page">
    <div class="referral-tree-container">
        
        <!-- Referral Tree Header -->
        <div class="referral-header">
            <h2>Referral Tree</h2>
            <p>Track your referral network and monitor their performance</p>
            
            <!-- Referral Link Section -->
            <div class="referral-link-section">
                <div class="referral-link-card">
                    <h4>Your Referral Link</h4>
                    <div class="link-container">
                        <input type="text" id="referralLink" value="{{ $getReferral->direct_link ?? '#' }}" readonly>
                        <button class="copy-btn" onclick="copyReferralLink()">
                            <i class="fas fa-copy"></i>
                        </button>
                    </div>
                    <p class="link-note">Share this link to earn commissions from your referrals</p>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-users"></i>
                </div>
                <div class="stat-content">
                    <h3>{{ $totalReferrals }}</h3>
                    <p>Total Referrals</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-user-check"></i>
                </div>
                <div class="stat-content">
                    <h3>{{ $activeReferrals }}</h3>
                    <p>Active Referrals</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-chart-line"></i>
                </div>
                <div class="stat-content">
                    <h3>{{ $currency }}{{ number_format($totalInvestment, 0) }}</h3>
                    <p>Total Investment</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-dollar-sign"></i>
                </div>
                <div class="stat-content">
                    <h3>{{ $currency }}{{ number_format($totalEarnings, 0) }}</h3>
                    <p>Total Earnings</p>
                </div>
            </div>
        </div>

        <!-- Referral Levels -->
        <div class="referral-levels">
            <h3>Referral Levels</h3>
            <div class="levels-grid">
                @for($i = 1; $i <= 10; $i++)
                    <div class="level-card">
                        <div class="level-header">
                            <h4>Level {{ $i }}</h4>
                            <span class="level-count">{{ $referralLevels[$i]['count'] }} Users</span>
                        </div>
                        <div class="level-stats">
                            <div class="level-stat">
                                <span class="label">Investment:</span>
                                <span class="value">{{ $currency }}{{ number_format($referralLevels[$i]['investment'], 0) }}</span>
                            </div>
                            <div class="level-stat">
                                <span class="label">Deposit:</span>
                                <span class="value">{{ $currency }}{{ number_format($referralLevels[$i]['deposit'], 0) }}</span>
                            </div>
                            <div class="level-stat">
                                <span class="label">Profit:</span>
                                <span class="value">{{ $currency }}{{ number_format($referralLevels[$i]['profit'], 0) }}</span>
                            </div>
                        </div>
                    </div>
                @endfor
            </div>
        </div>

        <!-- Referral Tree -->
        <div class="referral-tree-section">
            <div class="tree-header">
                <h3>Referral Tree Structure</h3>
                <div class="zoom-controls">
                    <button class="zoom-btn" id="zoomOutBtn" title="Zoom Out">
                        <i class="fas fa-minus"></i>
                    </button>
                    <span class="zoom-level" id="zoomLevel">100%</span>
                    <button class="zoom-btn" id="zoomInBtn" title="Zoom In">
                        <i class="fas fa-plus"></i>
                    </button>
                    <button class="zoom-btn" id="resetZoomBtn" title="Reset Zoom">
                        <i class="fas fa-expand-arrows-alt"></i>
                    </button>
                </div>
            </div>
            <div class="tree-container">
                <div class="tree-wrapper" id="treeWrapper">
                    <!-- Root User -->
                    <div class="tree-node root-node">
                        <div class="node-content">
                            <div class="node-avatar">
                                <i class="fas fa-user"></i>
                            </div>
                            <div class="node-info">
                                <h4>{{ $user->username }}</h4>
                                <p>{{ $user->email }}</p>
                                <span class="node-status active">You</span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Referral Tree -->
                    @if($referralTree->count() > 0)
                        <div class="tree-children">
                            @foreach($referralTree as $referral)
                                @include('frontend.shahdeveloper.referral-tree.partials.tree-node', ['referral' => $referral, 'level' => 1])
                            @endforeach
                        </div>
                    @else
                        <div class="no-referrals">
                            <i class="fas fa-users"></i>
                            <h4>No Referrals Yet</h4>
                            <p>Start referring users to build your network</p>
                        </div>
                    @endif
                </div>
            </div>
        </div>

        <!-- Referral List -->
        <div class="referral-list-section">
            <h3>All Referrals</h3>
            <div class="referral-list-container">
                <div class="list-header">
                    <div class="search-box">
                        <i class="fas fa-search"></i>
                        <input type="text" id="searchReferrals" placeholder="Search referrals...">
                    </div>
                    <div class="filter-buttons">
                        <button class="filter-btn active" data-filter="all">All</button>
                        <button class="filter-btn" data-filter="active">Active</button>
                        <button class="filter-btn" data-filter="inactive">Inactive</button>
                    </div>
                </div>
                
                <div class="referral-table-wrapper">
                    <table class="referral-table">
                        <thead>
                            <tr>
                                <th>User</th>
                                <th>Level</th>
                                <th>Investment</th>
                                <th>Deposit</th>
                                <th>Profit</th>
                                <th>Referrals</th>
                                <th>Status</th>
                                <th>Join Date</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody id="referralTableBody">
                            @foreach($directReferrals as $referral)
                                <tr class="referral-row" data-level="1" data-status="{{ $referral->status ? 'active' : 'inactive' }}">
                                    <td>
                                        <div class="user-info">
                                            <div class="user-avatar">
                                                <i class="fas fa-user"></i>
                                            </div>
                                            <div class="user-details">
                                                <h4>{{ $referral->username }}</h4>
                                                <p>{{ $referral->email }}</p>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <span class="level-badge">Level 1</span>
                                    </td>
                                    <td>{{ $currency }}{{ number_format($referral->totalInvestment(), 0) }}</td>
                                    <td>{{ $currency }}{{ number_format($referral->totalDeposit(), 0) }}</td>
                                    <td>{{ $currency }}{{ number_format($referral->totalProfit(), 0) }}</td>
                                    <td>{{ $referral->referrals_count }}</td>
                                    <td>
                                        <span class="status-badge {{ $referral->status ? 'active' : 'inactive' }}">
                                            {{ $referral->status ? 'Active' : 'Inactive' }}
                                        </span>
                                    </td>
                                    <td>{{ $referral->created_at->format('M d, Y') }}</td>
                                    <td>
                                        <button class="btn-view" onclick="viewReferralDetails({{ $referral->id }})">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>
</div>

<!-- Referral Details Modal -->
<div id="referralModal" class="modal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Referral Details</h3>
            <button class="close-btn" onclick="closeModal()">&times;</button>
        </div>
        <div class="modal-body" id="referralModalBody">
            <!-- Content will be loaded here -->
        </div>
    </div>
</div>

@endsection

@push('style')
<style>
/* Referral Tree Page Styles */
.referral-tree-page {
    padding: 20px;
    background: transparent;
    min-height: 100vh;
}

.referral-tree-container {
    display: flex;
    flex-direction: column;
    gap: 25px;
}

.referral-header {
    text-align: center;
    margin-bottom: 25px;
    padding: 25px;
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(16px);
    -webkit-backdrop-filter: blur(16px);
    border-radius: 16px;
    border: 1px solid rgba(0, 255, 136, 0.18);
    box-shadow: 0 10px 30px rgba(0, 255, 136, 0.12);
    position: relative;
    overflow: hidden;
}

.referral-header::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #00ff88, #00e0a3);
}

.referral-header h2 {
    font-size: 1.8rem;
    font-weight: 700;
    color: #e8fff5;
    margin-bottom: 10px;
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.referral-header p {
    font-size: 1rem;
    color: #a7c8bb;
    margin: 0 0 20px 0;
}

/* Referral Link Section */
.referral-link-section {
    margin-top: 20px;
}

.referral-link-card {
    background: rgba(0, 255, 136, 0.06);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    padding: 20px;
    text-align: center;
}

.referral-link-card h4 {
    font-size: 1.1rem;
    font-weight: 600;
    color: #e8fff5;
    margin: 0 0 15px 0;
}

.link-container {
    display: flex;
    gap: 10px;
    margin-bottom: 10px;
}

.link-container input {
    flex: 1;
    padding: 12px 15px;
    background: #101a16;
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 12px;
    color: #e8fff5;
    font-size: 0.9rem;
    font-family: 'Courier New', monospace;
}

.link-container input:focus {
    outline: none;
    border-color: #00ff88;
}

.copy-btn {
    background: #00ff88;
    color: #0a0f0d;
    border: none;
    padding: 12px 15px;
    border-radius: 12px;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    font-size: 0.9rem;
}

.copy-btn:hover {
    background: #00e0a3;
    transform: translateY(-1px);
}

.link-note {
    font-size: 0.85rem;
    color: #88a498;
    margin: 0;
}

/* Statistics Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 25px;
}

.stat-card {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(16px);
    -webkit-backdrop-filter: blur(16px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    padding: 25px;
    display: flex;
    align-items: center;
    gap: 20px;
    box-shadow: 0 10px 30px rgba(0, 255, 136, 0.12);
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, #00ff88, transparent);
    opacity: 0.6;
}

.stat-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 25px 60px rgba(0, 255, 136, 0.18);
}

.stat-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
    color: #0a0f0d;
    flex-shrink: 0;
}

.stat-content h3 {
    font-size: 1.8rem;
    font-weight: 700;
    color: #e8fff5;
    margin: 0 0 5px 0;
}

.stat-content p {
    font-size: 0.9rem;
    color: #a7c8bb;
    margin: 0;
}

/* Referral Levels */
.referral-levels {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(16px);
    -webkit-backdrop-filter: blur(16px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 10px 30px rgba(0, 255, 136, 0.12);
    position: relative;
    overflow: hidden;
}

.referral-levels::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #00ff88, #00e0a3);
}

.referral-levels h3 {
    font-size: 1.4rem;
    font-weight: 700;
    color: #e8fff5;
    margin-bottom: 20px;
}

.levels-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
}

.level-card {
    background: rgba(255, 255, 255, 0.03);
    backdrop-filter: blur(12px);
    -webkit-backdrop-filter: blur(12px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 12px;
    padding: 20px;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    position: relative;
    overflow: hidden;
}

.level-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, #00ff88, transparent);
    opacity: 0.4;
}

.level-card:hover {
    background: rgba(0, 255, 136, 0.12);
    transform: translateY(-2px);
}

.level-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
}

.level-header h4 {
    font-size: 1.1rem;
    font-weight: 600;
    color: #e8fff5;
    margin: 0;
}

.level-count {
    background: #00ff88;
    color: #0a0f0d;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8rem;
    font-weight: 600;
}

.level-stats {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.level-stat {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.level-stat .label {
    font-size: 0.85rem;
    color: #a7c8bb;
}

.level-stat .value {
    font-size: 0.85rem;
    font-weight: 600;
    color: #e8fff5;
}

/* Referral Tree Section */
.referral-tree-section {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(16px);
    -webkit-backdrop-filter: blur(16px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 10px 30px rgba(0, 255, 136, 0.12);
    position: relative;
    overflow: hidden;
}

.referral-tree-section::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #00ff88, #00e0a3);
}

.tree-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 15px;
}

.tree-header h3 {
    margin: 0;
    color: #e8fff5;
    font-size: 1.3rem;
    font-weight: 600;
}

.zoom-controls {
    display: flex;
    align-items: center;
    gap: 10px;
    background: #0a0f0d;
    padding: 8px 12px;
    border-radius: 12px;
    border: 1px solid rgba(0, 255, 136, 0.18);
}

.zoom-btn {
    background: #101a16;
    color: #e8fff5;
    border: 1px solid rgba(0, 255, 136, 0.18);
    padding: 8px 12px;
    border-radius: 12px;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    justify-content: center;
    min-width: 36px;
    height: 36px;
}

.zoom-btn:hover {
    background: #00ff88;
    color: #0a0f0d;
    border-color: #00ff88;
    transform: translateY(-1px);
}

.zoom-level {
    color: #e8fff5;
    font-weight: 600;
    font-size: 0.9rem;
    min-width: 45px;
    text-align: center;
}

.tree-container {
    overflow: auto;
    padding: 20px 0;
    max-height: 600px;
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 12px;
    background: #0a0f0d;
}

#treeWrapper {
    transition: transform 0.3s ease;
    transform-origin: top left;
    min-width: 100%;
    min-height: 100%;
}

.tree-wrapper {
    min-width: 600px;
    position: relative;
}

.tree-node {
    display: flex;
    flex-direction: column;
    align-items: center;
    margin-bottom: 30px;
    position: relative;
}

.root-node {
    margin-bottom: 40px;
}

.node-content {
    background: rgba(0, 255, 136, 0.06);
    border: 2px solid #00ff88;
    border-radius: 16px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    min-width: 250px;
    box-shadow: 0 4px 15px rgba(0, 255, 136, 0.2);
}

.node-avatar {
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, #00ff88, #00e0a3);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
    color: #0a0f0d;
    flex-shrink: 0;
}

.node-info h4 {
    font-size: 1rem;
    font-weight: 600;
    color: #e8fff5;
    margin: 0 0 5px 0;
}

.node-info p {
    font-size: 0.8rem;
    color: #a7c8bb;
    margin: 0 0 5px 0;
}

.node-status {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.75rem;
    font-weight: 600;
}

.node-status.active {
    background: rgba(0, 255, 136, 0.2);
    color: #00ff88;
}

.node-status.inactive {
    background: rgba(255, 255, 255, 0.1);
    color: #88a498;
}

.tree-children {
    display: flex;
    flex-wrap: wrap;
    gap: 20px;
    justify-content: center;
}

.no-referrals {
    text-align: center;
    padding: 40px;
    color: #88a498;
}

.no-referrals i {
    font-size: 3rem;
    margin-bottom: 15px;
    display: block;
}

.no-referrals h4 {
    font-size: 1.2rem;
    margin-bottom: 10px;
}

/* Referral List Section */
.referral-list-section {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(16px);
    -webkit-backdrop-filter: blur(16px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    padding: 25px;
    box-shadow: 0 10px 30px rgba(0, 255, 136, 0.12);
    position: relative;
    overflow: hidden;
}

.referral-list-section::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #00ff88, #00e0a3);
}

.referral-list-section h3 {
    font-size: 1.4rem;
    font-weight: 700;
    color: #e8fff5;
    margin-bottom: 20px;
}

.list-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    flex-wrap: wrap;
    gap: 15px;
}

.search-box {
    position: relative;
    flex: 1;
    max-width: 300px;
}

.search-box i {
    position: absolute;
    left: 12px;
    top: 50%;
    transform: translateY(-50%);
    color: #88a498;
}

.search-box input {
    width: 100%;
    padding: 10px 12px 10px 35px;
    background: rgba(0, 255, 136, 0.06);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 12px;
    color: #e8fff5;
    font-size: 0.9rem;
}

.search-box input:focus {
    outline: none;
    border-color: #00ff88;
}

.filter-buttons {
    display: flex;
    gap: 10px;
}

.filter-btn {
    padding: 8px 16px;
    background: rgba(0, 255, 136, 0.06);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 12px;
    color: #a7c8bb;
    font-size: 0.9rem;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.filter-btn.active,
.filter-btn:hover {
    background: #00ff88;
    color: #0a0f0d;
    border-color: #00ff88;
}

.referral-table-wrapper {
    overflow-x: auto;
    border-radius: 12px;
    background: rgba(255, 255, 255, 0.03);
    backdrop-filter: blur(12px);
    -webkit-backdrop-filter: blur(12px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    position: relative;
    overflow: hidden;
}

.referral-table-wrapper::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 1px;
    background: linear-gradient(90deg, transparent, #00ff88, transparent);
    opacity: 0.4;
}

.referral-table {
    width: 100%;
    border-collapse: collapse;
    margin: 0;
    min-width: 800px;
}

.referral-table th {
    background: linear-gradient(135deg, rgba(0, 255, 136, 0.12), rgba(0, 255, 136, 0.06));
    color: #e8fff5;
    font-weight: 700;
    padding: 15px 12px;
    text-align: left;
    border-bottom: 2px solid rgba(0, 255, 136, 0.18);
    font-size: 0.9rem;
}

.referral-table td {
    padding: 15px 12px;
    border-bottom: 1px solid rgba(0, 255, 136, 0.18);
    vertical-align: middle;
    background: rgba(255, 255, 255, 0.02);
    backdrop-filter: blur(8px);
    -webkit-backdrop-filter: blur(8px);
}

.user-info {
    display: flex;
    align-items: center;
    gap: 12px;
}

.user-avatar {
    width: 40px;
    height: 40px;
    background: rgba(0, 255, 136, 0.06);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 16px;
    color: #a7c8bb;
}

.user-details h4 {
    font-size: 0.9rem;
    font-weight: 600;
    color: #e8fff5;
    margin: 0 0 2px 0;
}

.user-details p {
    font-size: 0.8rem;
    color: #a7c8bb;
    margin: 0;
}

.level-badge {
    background: #00ff88;
    color: #0a0f0d;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8rem;
    font-weight: 600;
}

.status-badge {
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 0.8rem;
    font-weight: 600;
}

.status-badge.active {
    background: rgba(0, 255, 136, 0.2);
    color: #00ff88;
}

.status-badge.inactive {
    background: rgba(255, 255, 255, 0.1);
    color: #88a498;
}

.btn-view {
    background: #00ff88;
    color: #0a0f0d;
    border: none;
    padding: 6px 10px;
    border-radius: 12px;
    font-size: 0.8rem;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.btn-view:hover {
    background: #00e0a3;
    transform: translateY(-1px);
}

/* Modal Styles */
.modal {
    display: none;
    position: fixed;
    z-index: 1000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(5px);
}

.modal-content {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    -webkit-backdrop-filter: blur(20px);
    border: 1px solid rgba(0, 255, 136, 0.18);
    border-radius: 16px;
    margin: 5% auto;
    padding: 0;
    width: 90%;
    max-width: 600px;
    box-shadow: 0 25px 60px rgba(0, 255, 136, 0.18);
    position: relative;
    overflow: hidden;
}

.modal-content::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 2px;
    background: linear-gradient(90deg, #00ff88, #00e0a3);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 25px;
    border-bottom: 1px solid rgba(0, 255, 136, 0.18);
}

.modal-header h3 {
    font-size: 1.3rem;
    font-weight: 700;
    color: #e8fff5;
    margin: 0;
}

.close-btn {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #88a498;
    cursor: pointer;
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.close-btn:hover {
    color: #e8fff5;
}

.modal-body {
    padding: 25px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .referral-tree-page {
        padding: 15px;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .stat-card {
        padding: 20px;
        flex-direction: column;
        text-align: center;
        gap: 15px;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
        font-size: 20px;
    }
    
    .levels-grid {
        grid-template-columns: 1fr;
    }
    
    .level-card {
        padding: 15px;
    }
    
    .list-header {
        flex-direction: column;
        align-items: stretch;
        gap: 15px;
    }
    
    .search-box {
        max-width: none;
    }
    
    .filter-buttons {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .tree-header {
        flex-direction: column;
        align-items: stretch;
        gap: 10px;
    }
    
    .zoom-controls {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .zoom-btn {
        min-width: 32px;
        height: 32px;
        font-size: 0.8rem;
    }
    
    .tree-container {
        max-height: 400px;
    }
    
    .referral-table {
        min-width: 600px;
    }
    
    .node-content {
        min-width: 200px;
        flex-direction: column;
        text-align: center;
        padding: 15px;
    }
    
    .tree-children {
        flex-direction: column;
        align-items: center;
    }
    
    .referral-header {
        padding: 20px;
    }
    
    .referral-header h2 {
        font-size: 1.5rem;
    }
    
    .referral-link-card {
        padding: 15px;
    }
    
    .link-container {
        flex-direction: column;
        gap: 10px;
    }
    
    .link-container input {
        font-size: 0.8rem;
    }
    
    .copy-btn {
        padding: 10px 15px;
        font-size: 0.8rem;
    }
}

@media (max-width: 480px) {
    .referral-tree-page {
        padding: 10px;
    }
    
    .stat-card {
        padding: 15px;
    }
    
    .stat-content h3 {
        font-size: 1.5rem;
    }
    
    .level-card {
        padding: 12px;
    }
    
    .node-content {
        min-width: 180px;
        padding: 12px;
    }
    
    .referral-table {
        min-width: 500px;
    }
    
    .referral-table th,
    .referral-table td {
        padding: 10px 8px;
        font-size: 0.8rem;
    }
    
    .user-info {
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .user-avatar {
        width: 35px;
        height: 35px;
        font-size: 14px;
    }
    
    .modal-content {
        width: 95%;
        margin: 10% auto;
    }
    
    .modal-header {
        padding: 15px 20px;
    }
    
    .modal-body {
        padding: 20px;
    }
}
</style>
@endpush

@push('script')
<script>
// Search functionality
document.getElementById('searchReferrals').addEventListener('input', function(e) {
    const searchTerm = e.target.value.toLowerCase();
    const rows = document.querySelectorAll('.referral-row');
    
    rows.forEach(row => {
        const username = row.querySelector('.user-details h4').textContent.toLowerCase();
        const email = row.querySelector('.user-details p').textContent.toLowerCase();
        
        if (username.includes(searchTerm) || email.includes(searchTerm)) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
});

// Filter functionality
document.querySelectorAll('.filter-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        // Remove active class from all buttons
        document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
        // Add active class to clicked button
        this.classList.add('active');
        
        const filter = this.dataset.filter;
        const rows = document.querySelectorAll('.referral-row');
        
        rows.forEach(row => {
            if (filter === 'all') {
                row.style.display = '';
            } else {
                const status = row.dataset.status;
                if (status === filter) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            }
        });
    });
});

// View referral details
function viewReferralDetails(referralId) {
    fetch(`/user/referral-tree/details?referral_id=${referralId}`)
        .then(response => response.json())
        .then(data => {
            if (data.error) {
                alert(data.error);
                return;
            }
            
            const modalBody = document.getElementById('referralModalBody');
            modalBody.innerHTML = `
                <div class="referral-details">
                    <div class="detail-header">
                        <div class="detail-avatar">
                            <i class="fas fa-user"></i>
                        </div>
                        <div class="detail-info">
                            <h4>${data.username}</h4>
                            <p>${data.email}</p>
                            <span class="status-badge ${data.status ? 'active' : 'inactive'}">${data.status_text}</span>
                        </div>
                    </div>
                    
                    <div class="detail-stats">
                        <div class="detail-stat">
                            <span class="label">Investment:</span>
                            <span class="value">$${data.total_investment}</span>
                        </div>
                        <div class="detail-stat">
                            <span class="label">Deposit:</span>
                            <span class="value">$${data.total_deposit}</span>
                        </div>
                        <div class="detail-stat">
                            <span class="label">Profit:</span>
                            <span class="value">$${data.total_profit}</span>
                        </div>
                        <div class="detail-stat">
                            <span class="label">Referrals:</span>
                            <span class="value">${data.referral_count}</span>
                        </div>
                        <div class="detail-stat">
                            <span class="label">Join Date:</span>
                            <span class="value">${data.join_date}</span>
                        </div>
                    </div>
                </div>
            `;
            
            document.getElementById('referralModal').style.display = 'block';
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Error loading referral details');
        });
}

// Close modal
function closeModal() {
    document.getElementById('referralModal').style.display = 'none';
}

// Close modal when clicking outside
window.onclick = function(event) {
    const modal = document.getElementById('referralModal');
    if (event.target == modal) {
        modal.style.display = 'none';
    }
}

// Copy referral link
function copyReferralLink() {
    const referralLink = document.getElementById('referralLink');
    referralLink.select();
    referralLink.setSelectionRange(0, 99999); // For mobile devices
    
    try {
        document.execCommand('copy');
        
        // Show success message
        const copyBtn = document.querySelector('.copy-btn');
        const originalText = copyBtn.innerHTML;
        copyBtn.innerHTML = '<i class="fas fa-check"></i>';
        copyBtn.style.background = '#00ff88';
        
        setTimeout(() => {
            copyBtn.innerHTML = originalText;
            copyBtn.style.background = '#00ff88';
        }, 2000);
        
    } catch (err) {
        console.error('Failed to copy: ', err);
        alert('Failed to copy link');
    }
}

// Zoom functionality - Global scope
window.currentZoom = 1;
window.minZoom = 0.5;
window.maxZoom = 3;
window.zoomStep = 0.1;

window.zoomIn = function() {
    console.log('Zoom In clicked, current zoom:', window.currentZoom);
    if (window.currentZoom < window.maxZoom) {
        window.currentZoom = Math.min(window.currentZoom + window.zoomStep, window.maxZoom);
        console.log('New zoom level:', window.currentZoom);
        window.applyZoom();
    } else {
        console.log('Already at max zoom');
    }
}

window.zoomOut = function() {
    console.log('Zoom Out clicked, current zoom:', window.currentZoom);
    if (window.currentZoom > window.minZoom) {
        window.currentZoom = Math.max(window.currentZoom - window.zoomStep, window.minZoom);
        console.log('New zoom level:', window.currentZoom);
        window.applyZoom();
    } else {
        console.log('Already at min zoom');
    }
}

window.resetZoom = function() {
    console.log('Reset Zoom clicked');
    window.currentZoom = 1;
    window.applyZoom();
}

window.applyZoom = function() {
    const treeWrapper = document.getElementById('treeWrapper');
    const zoomLevel = document.getElementById('zoomLevel');
    
    console.log('Applying zoom:', window.currentZoom);
    console.log('Tree wrapper found:', !!treeWrapper);
    console.log('Zoom level found:', !!zoomLevel);
    
    if (treeWrapper && zoomLevel) {
        treeWrapper.style.transform = `scale(${window.currentZoom})`;
        zoomLevel.textContent = Math.round(window.currentZoom * 100) + '%';
        console.log('Zoom applied successfully');
    } else {
        console.error('Tree wrapper or zoom level element not found');
    }
}

// Touch/pinch zoom support for mobile
let initialDistance = 0;
let initialZoom = 1;

document.addEventListener('touchstart', function(e) {
    if (e.touches.length === 2) {
        e.preventDefault();
        initialDistance = Math.hypot(
            e.touches[0].clientX - e.touches[1].clientX,
            e.touches[0].clientY - e.touches[1].clientY
        );
        initialZoom = window.currentZoom;
    }
});

document.addEventListener('touchmove', function(e) {
    if (e.touches.length === 2) {
        e.preventDefault();
        const currentDistance = Math.hypot(
            e.touches[0].clientX - e.touches[1].clientX,
            e.touches[0].clientY - e.touches[1].clientY
        );
        
        if (initialDistance > 0) {
            const scale = currentDistance / initialDistance;
            const newZoom = initialZoom * scale;
            
            if (newZoom >= window.minZoom && newZoom <= window.maxZoom) {
                window.currentZoom = newZoom;
                window.applyZoom();
            }
        }
    }
});

document.addEventListener('touchend', function(e) {
    if (e.touches.length < 2) {
        initialDistance = 0;
        initialZoom = 1;
    }
});

// Initialize zoom on page load
document.addEventListener('DOMContentLoaded', function() {
    console.log('Page loaded, initializing zoom...');
    window.applyZoom();
    
    // Add event listeners for zoom buttons
    const zoomInBtn = document.getElementById('zoomInBtn');
    const zoomOutBtn = document.getElementById('zoomOutBtn');
    const resetZoomBtn = document.getElementById('resetZoomBtn');
    
    if (zoomInBtn) {
        zoomInBtn.addEventListener('click', function() {
            console.log('Zoom In button clicked');
            window.zoomIn();
        });
    }
    
    if (zoomOutBtn) {
        zoomOutBtn.addEventListener('click', function() {
            console.log('Zoom Out button clicked');
            window.zoomOut();
        });
    }
    
    if (resetZoomBtn) {
        resetZoomBtn.addEventListener('click', function() {
            console.log('Reset Zoom button clicked');
            window.resetZoom();
        });
    }
    
    console.log('Event listeners added successfully');
});
</script>

               