@extends('frontend.shahdeveloper.layouts.user')
@section('title')
ROI Collection
@endsection
@section('subtitle')
Collect your ready ROI and track your investment returns
@endsection
@section('content')

<div class="roi-collection-page">
    <div class="roi-container">
        
        <!-- ROI Statistics Cards -->
        <div class="roi-stats-grid">
            <div class="stat-card pending">
                <div class="stat-icon">
                    <i class="fas fa-clock"></i>
                </div>
                <div class="stat-content">
                    <h3>Pending ROI</h3>
                    <p class="stat-amount">${{ number_format($statistics['pending'], 2) }}</p>
                    <small>Waiting for timer</small>
                </div>
            </div>
            
            <div class="stat-card ready">
                <div class="stat-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <div class="stat-content">
                    <h3>Ready to Collect</h3>
                    <p class="stat-amount">${{ number_format($statistics['ready'], 2) }}</p>
                    <small>Available now</small>
                </div>
            </div>
            
            <div class="stat-card collected">
                <div class="stat-icon">
                    <i class="fas fa-wallet"></i>
                </div>
                <div class="stat-content">
                    <h3>Total Collected</h3>
                    <p class="stat-amount">${{ number_format($statistics['collected'], 2) }}</p>
                    <small>In your wallet</small>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        @if($statistics['ready'] > 0)
        <div class="quick-actions">
            <button class="btn btn-primary collect-all-btn" onclick="collectAllRoi()">
                <i class="fas fa-download"></i>
                Collect All Ready ROI (${{ number_format($statistics['ready'], 2) }})
            </button>
        </div>
        @endif

        <!-- ROI Schedules Table -->
        <div class="roi-schedules-section">
            <div class="section-header">
                <h3>ROI Schedules</h3>
                <div class="filter-tabs">
                    <button class="filter-btn active" data-status="all">All</button>
                    <button class="filter-btn" data-status="pending">Pending</button>
                    <button class="filter-btn" data-status="ready">Ready</button>
                    <button class="filter-btn" data-status="sent">Collected</button>
                </div>
            </div>

            <div class="roi-table-container">
                <table class="roi-table">
                    <thead>
                        <tr>
                            <th>Investment Plan</th>
                            <th>Cycle</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Timer</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($roiSchedules as $schedule)
                        <tr class="roi-row" data-status="{{ $schedule->status }}">
                            <td>
                                <div class="plan-info">
                                    <strong>{{ $schedule->invest->schema->name ?? 'N/A' }}</strong>
                                    <small>Investment: ${{ number_format($schedule->invest_amount, 2) }}</small>
                                </div>
                            </td>
                            <td>
                                <span class="cycle-badge">
                                    {{ $schedule->roi_cycle }}/{{ $schedule->total_cycles }}
                                </span>
                            </td>
                            <td>
                                <span class="amount">${{ number_format($schedule->roi_amount, 2) }}</span>
                            </td>
                            <td>
                                <span class="status-badge status-{{ $schedule->status }}">
                                    {{ ucfirst($schedule->status) }}
                                </span>
                            </td>
                            <td>
                                @if($schedule->status === 'pending')
                                    <div class="timer" data-end-time="{{ $schedule->roi_end_time->timestamp }}">
                                        <span class="timer-text">Loading...</span>
                                    </div>
                                @elseif($schedule->status === 'ready')
                                    <span class="ready-text">Ready Now!</span>
                                @else
                                    <span class="collected-text">
                                        Collected {{ $schedule->roi_sent_time ? $schedule->roi_sent_time->format('M d, Y H:i') : 'N/A' }}
                                    </span>
                                @endif
                            </td>
                            <td>
                                @if($schedule->status === 'ready')
                                    <button class="btn btn-sm btn-success collect-btn" 
                                            onclick="collectRoi({{ $schedule->id }})">
                                        <i class="fas fa-download"></i>
                                        Collect
                                    </button>
                                @elseif($schedule->status === 'sent')
                                    <span class="collected-icon">
                                        <i class="fas fa-check-circle"></i>
                                    </span>
                                @else
                                    <span class="waiting-text">Waiting...</span>
                                @endif
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="6" class="text-center">
                                <div class="empty-state">
                                    <i class="fas fa-chart-line"></i>
                                    <h4>No ROI Schedules Found</h4>
                                    <p>Start investing to see your ROI schedules here.</p>
                                </div>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            <div class="pagination-wrapper">
                {{ $roiSchedules->links() }}
            </div>
        </div>
    </div>
</div>

<style>
/* ROI Collection Styles */
.roi-collection-page {
    padding: 20px;
    max-width: 1200px;
    margin: 0 auto;
}

.roi-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: linear-gradient(135deg, var(--surface), var(--surface-light));
    border: 1px solid var(--border);
    border-radius: var(--border-radius);
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    transition: var(--transition);
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow);
}

.stat-card.pending {
    border-left: 4px solid var(--warning);
}

.stat-card.ready {
    border-left: 4px solid var(--success);
}

.stat-card.collected {
    border-left: 4px solid var(--primary-color);
}

.stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 20px;
}

.stat-card.pending .stat-icon {
    background: rgba(217, 119, 6, 0.2);
    color: var(--warning);
}

.stat-card.ready .stat-icon {
    background: rgba(5, 150, 105, 0.2);
    color: var(--success);
}

.stat-card.collected .stat-icon {
    background: rgba(30, 64, 175, 0.2);
    color: var(--primary-color);
}

.stat-content h3 {
    margin: 0 0 5px 0;
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 500;
}

.stat-amount {
    font-size: 24px;
    font-weight: 700;
    color: var(--text-primary);
    margin: 0 0 5px 0;
}

.stat-content small {
    color: var(--text-muted);
    font-size: 12px;
}

.quick-actions {
    margin-bottom: 30px;
    text-align: center;
}

.collect-all-btn {
    background: linear-gradient(135deg, var(--success), #10b981);
    color: white;
    border: none;
    padding: 15px 30px;
    border-radius: var(--border-radius);
    font-weight: 600;
    font-size: 16px;
    cursor: pointer;
    transition: var(--transition);
    box-shadow: 0 4px 15px rgba(5, 150, 105, 0.3);
}

.collect-all-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 6px 20px rgba(5, 150, 105, 0.4);
}

.roi-schedules-section {
    background: var(--surface);
    border: 1px solid var(--border);
    border-radius: var(--border-radius);
    overflow: hidden;
}

.section-header {
    padding: 20px;
    border-bottom: 1px solid var(--border);
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 15px;
}

.section-header h3 {
    margin: 0;
    color: var(--text-primary);
}

.filter-tabs {
    display: flex;
    gap: 5px;
}

.filter-btn {
    padding: 8px 16px;
    border: 1px solid var(--border);
    background: transparent;
    color: var(--text-secondary);
    border-radius: var(--border-radius-sm);
    cursor: pointer;
    transition: var(--transition);
    font-size: 14px;
}

.filter-btn.active,
.filter-btn:hover {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.roi-table-container {
    overflow-x: auto;
}

.roi-table {
    width: 100%;
    border-collapse: collapse;
}

.roi-table th {
    background: var(--background-secondary);
    color: var(--text-secondary);
    padding: 15px;
    text-align: left;
    font-weight: 600;
    font-size: 14px;
    border-bottom: 1px solid var(--border);
}

.roi-table td {
    padding: 15px;
    border-bottom: 1px solid var(--border);
    vertical-align: middle;
}

.roi-row:hover {
    background: var(--surface-light);
}

.plan-info strong {
    display: block;
    color: var(--text-primary);
    margin-bottom: 5px;
}

.plan-info small {
    color: var(--text-muted);
    font-size: 12px;
}

.cycle-badge {
    background: var(--primary-color);
    color: white;
    padding: 4px 8px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 600;
}

.amount {
    font-weight: 700;
    color: var(--success);
    font-size: 16px;
}

.status-badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-pending {
    background: rgba(217, 119, 6, 0.2);
    color: var(--warning);
}

.status-ready {
    background: rgba(5, 150, 105, 0.2);
    color: var(--success);
}

.status-sent {
    background: rgba(30, 64, 175, 0.2);
    color: var(--primary-color);
}

.timer {
    font-family: 'Courier New', monospace;
}

.timer-text {
    color: var(--warning);
    font-weight: 600;
}

.ready-text {
    color: var(--success);
    font-weight: 600;
}

.collected-text {
    color: var(--text-muted);
    font-size: 12px;
}

.collect-btn {
    background: var(--success);
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: var(--border-radius-sm);
    cursor: pointer;
    transition: var(--transition);
    font-size: 12px;
    font-weight: 600;
}

.collect-btn:hover {
    background: #10b981;
    transform: translateY(-1px);
}

.collected-icon {
    color: var(--success);
    font-size: 18px;
}

.waiting-text {
    color: var(--text-muted);
    font-size: 12px;
}

.empty-state {
    text-align: center;
    padding: 40px 20px;
    color: var(--text-muted);
}

.empty-state i {
    font-size: 48px;
    margin-bottom: 15px;
    color: var(--text-muted);
}

.empty-state h4 {
    margin: 0 0 10px 0;
    color: var(--text-secondary);
}

.empty-state p {
    margin: 0;
    font-size: 14px;
}

.pagination-wrapper {
    padding: 20px;
    text-align: center;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .roi-collection-page {
        padding: 15px;
    }
    
    .roi-stats-grid {
        grid-template-columns: 1fr;
        gap: 15px;
    }
    
    .section-header {
        flex-direction: column;
        align-items: stretch;
    }
    
    .filter-tabs {
        justify-content: center;
    }
    
    .roi-table {
        font-size: 14px;
    }
    
    .roi-table th,
    .roi-table td {
        padding: 10px 8px;
    }
}
</style>

<script>
// Timer functionality
function updateTimers() {
    document.querySelectorAll('.timer').forEach(timer => {
        const endTime = parseInt(timer.dataset.endTime);
        const now = Math.floor(Date.now() / 1000);
        const remaining = endTime - now;
        
        if (remaining > 0) {
            const hours = Math.floor(remaining / 3600);
            const minutes = Math.floor((remaining % 3600) / 60);
            const seconds = remaining % 60;
            
            timer.querySelector('.timer-text').textContent = 
                `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
        } else {
            timer.querySelector('.timer-text').textContent = 'Ready!';
            timer.parentElement.parentElement.querySelector('.status-badge').textContent = 'Ready';
            timer.parentElement.parentElement.querySelector('.status-badge').className = 'status-badge status-ready';
        }
    });
}

// Update timers every second
setInterval(updateTimers, 1000);
updateTimers();

// Filter functionality
document.querySelectorAll('.filter-btn').forEach(btn => {
    btn.addEventListener('click', () => {
        document.querySelectorAll('.filter-btn').forEach(b => b.classList.remove('active'));
        btn.classList.add('active');
        
        const status = btn.dataset.status;
        document.querySelectorAll('.roi-row').forEach(row => {
            if (status === 'all' || row.dataset.status === status) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    });
});

// Collect single ROI
function collectRoi(scheduleId) {
    if (confirm('Are you sure you want to collect this ROI?')) {
        fetch(`/user/roi/collect`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({ schedule_id: scheduleId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('ROI collected successfully! Amount: $' + data.amount);
                location.reload();
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while collecting ROI');
        });
    }
}

// Collect all ROI
function collectAllRoi() {
    if (confirm('Are you sure you want to collect all ready ROI?')) {
        fetch(`/user/roi/collect-all`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(`Successfully collected ${data.processed_count} ROI schedules! Total amount: $${data.total_amount}`);
                location.reload();
            } else {
                alert('Error: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('An error occurred while collecting ROI');
        });
    }
}
</script>

@endsection
