@extends('frontend.shahdeveloper.layouts.user')

@section('title', 'Withdrawal History')

@section('content')
<div class="withdraw-history-container">
    <div class="row">
        <div class="col-12">
            <!-- Page Header -->
            <div class="page-header">
                <div class="header-content">
                    <h1 class="page-title">
                        <i class="fas fa-history"></i>
                        Withdrawal History
                    </h1>
                    <p class="page-subtitle">Track all your withdrawal requests and their status</p>
                </div>
                <div class="header-actions">
                    <a href="{{ route('user.withdraw.simple.index') }}" class="btn btn-primary">
                        <i class="fas fa-plus"></i>
                        New Withdrawal
                    </a>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="stats-cards">
                @php
                    $pendingCount = $withdrawals ? $withdrawals->where('status', 'pending')->count() : 0;
                    $completedCount = $withdrawals ? $withdrawals->where('status', 'completed')->count() : 0;
                    $rejectedCount = $withdrawals ? $withdrawals->where('status', 'rejected')->count() : 0;
                    $totalWithdrawn = $withdrawals ? $withdrawals->where('status', 'completed')->sum('amount') : 0;
                @endphp
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-content">
                        <h3>{{ $pendingCount }}</h3>
                        <p>Pending</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3>{{ $completedCount }}</h3>
                        <p>Completed</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-times-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3>{{ $rejectedCount }}</h3>
                        <p>Rejected</p>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-dollar-sign"></i>
                    </div>
                    <div class="stat-content">
                        <h3>{{ $currencySymbol }}{{ number_format($totalWithdrawn, 2) }}</h3>
                        <p>Total Withdrawn</p>
                    </div>
                </div>
            </div>


            <!-- Withdrawals List -->
            <div class="withdrawals-section">
                @if($withdrawals && $withdrawals->count() > 0)
                    <div class="withdrawals-grid">
                        @foreach($withdrawals as $withdrawal)
                            <div class="withdrawal-card" data-status="{{ $withdrawal->status->value }}">
                                <div class="card-header">
                                    <div class="transaction-info">
                                        <h4 class="transaction-id">#{{ $withdrawal->id }}</h4>
                                        <p class="transaction-method">{{ $withdrawal->method ?? 'N/A' }}</p>
                                    </div>
                                    <div class="status-badge status-{{ $withdrawal->status->value }}">
                                        @switch($withdrawal->status->value)
                                            @case('pending')
                                                <i class="fas fa-clock"></i>
                                                Pending
                                                @break
                                            @case('approved')
                                                <i class="fas fa-check-circle"></i>
                                                Approved
                                                @break
                                            @case('completed')
                                                <i class="fas fa-check-double"></i>
                                                Completed
                                                @break
                                            @case('rejected')
                                                <i class="fas fa-times-circle"></i>
                                                Rejected
                                                @break
                                            @default
                                                <i class="fas fa-question-circle"></i>
                                                {{ ucfirst($withdrawal->status->value) }}
                                        @endswitch
                                    </div>
                                </div>
                                
                                <div class="card-body">
                                    <div class="amount-section">
                                        <div class="amount-display">
                                            <span class="currency">{{ $currencySymbol }}</span>
                                            <span class="amount">{{ number_format($withdrawal->amount, 2) }}</span>
                                        </div>
                                        @if($withdrawal->fee > 0)
                                            <div class="fee-info">
                                                <small>Fee: {{ $currencySymbol }}{{ number_format($withdrawal->fee, 2) }}</small>
                                            </div>
                                        @endif
                                        @if($withdrawal->final_amount != $withdrawal->amount)
                                            <div class="final-amount">
                                                <small>Final: {{ $currencySymbol }}{{ number_format($withdrawal->final_amount, 2) }}</small>
                                            </div>
                                        @endif
                                    </div>
                                    
                                    <div class="details-section">
                                        <div class="detail-row">
                                            <span class="label">Date:</span>
                                            <span class="value">{{ \Carbon\Carbon::parse($withdrawal->created_at)->format('M d, Y') }}</span>
                                        </div>
                                        <div class="detail-row">
                                            <span class="label">Time:</span>
                                            <span class="value">{{ \Carbon\Carbon::parse($withdrawal->created_at)->format('h:i A') }}</span>
                                        </div>
                                        <div class="detail-row">
                                            <span class="label">Type:</span>
                                            <span class="value">{{ ucfirst($withdrawal->type->value) }}</span>
                                        </div>
                                        <div class="detail-row">
                                            <span class="label">Wallet:</span>
                                            <span class="value">{{ ucfirst($withdrawal->details['wallet'] ?? 'main') }}</span>
                                        </div>
                                        @if($withdrawal->admin_note)
                                            <div class="detail-row">
                                                <span class="label">Admin Note:</span>
                                                <span class="value">{{ $withdrawal->admin_note }}</span>
                                            </div>
                                        @endif
                                        @if($withdrawal->rejection_reason)
                                            <div class="detail-row">
                                                <span class="label">Rejection Reason:</span>
                                                <span class="value">{{ $withdrawal->rejection_reason }}</span>
                                            </div>
                                        @endif
                                    </div>
                                </div>
                                
                                <div class="card-footer">
                                    <div class="action-buttons">
                                        <button class="btn btn-sm btn-outline-info copy-btn" onclick="copyToClipboard('{{ $withdrawal->id }}')">
                                            <i class="fas fa-copy"></i>
                                            Copy ID
                                        </button>
                                        @if($withdrawal->status->value === 'pending')
                                            <button class="btn btn-sm btn-outline-warning">
                                                <i class="fas fa-eye"></i>
                                                View Details
                                            </button>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>

                    <!-- Pagination -->
                    <div class="pagination-wrapper">
                        {{ $withdrawals ? $withdrawals->links() : '' }}
                    </div>
                @else
                    <!-- Empty State -->
                    <div class="empty-state">
                        <div class="empty-icon">
                            <i class="fas fa-money-bill-wave"></i>
                        </div>
                        <h3>No Withdrawals Yet</h3>
                        <p>You haven't made any withdrawal requests yet. Start by making your first withdrawal!</p>
                        <a href="{{ route('user.withdraw.simple.index') }}" class="btn btn-primary">
                            <i class="fas fa-plus"></i>
                            Make First Withdrawal
                        </a>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<style>
.withdraw-history-container {
    padding: 0 20px;
    max-width: 1200px;
    margin: 0 auto;
}

.page-header {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(0, 255, 136, 0.2);
    padding: 2rem;
    margin-bottom: 2rem;
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap;
    gap: 1rem;
}

.header-content h1.page-title {
    color: var(--text-primary);
    font-size: 2.5rem;
    font-weight: 700;
    margin: 0;
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    display: flex;
    align-items: center;
    gap: 1rem;
}

.header-content .page-subtitle {
    color: var(--text-secondary);
    font-size: 1.1rem;
    margin: 0.5rem 0 0 0;
}

.header-actions .btn {
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    color: #001f11;
    border: none;
    padding: 12px 24px;
    border-radius: 12px;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
}

.header-actions .btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 10px 20px rgba(0, 255, 136, 0.3);
    color: #001f11;
}

.stats-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    border-radius: 15px;
    border: 1px solid rgba(0, 255, 136, 0.2);
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    transition: all 0.3s ease;
}

.stat-card:hover {
    transform: translateY(-5px);
    border-color: rgba(0, 255, 136, 0.4);
}

.stat-icon {
    width: 60px;
    height: 60px;
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    border-radius: 15px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: #001f11;
}

.stat-content h3 {
    color: var(--text-primary);
    font-size: 2rem;
    font-weight: 700;
    margin: 0;
}

.stat-content p {
    color: var(--text-secondary);
    font-size: 1rem;
    margin: 0;
}

.withdrawals-section {
    background: rgba(255, 255, 255, 0.05);
    backdrop-filter: blur(20px);
    border-radius: 20px;
    border: 1px solid rgba(0, 255, 136, 0.2);
    padding: 2rem;
}

.withdrawals-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
    gap: 1.5rem;
}

.withdrawal-card {
    background: rgba(255, 255, 255, 0.03);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 15px;
    padding: 1.5rem;
    transition: all 0.3s ease;
}

.withdrawal-card:hover {
    border-color: rgba(0, 255, 136, 0.3);
    transform: translateY(-2px);
}

.card-header {
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    margin-bottom: 1rem;
}

.transaction-info h4.transaction-id {
    color: var(--text-primary);
    font-size: 1.1rem;
    font-weight: 600;
    margin: 0 0 0.25rem 0;
    font-family: 'Courier New', monospace;
}

.transaction-info p.transaction-method {
    color: var(--text-secondary);
    font-size: 0.9rem;
    margin: 0;
}

.status-badge {
    padding: 0.5rem 1rem;
    border-radius: 20px;
    font-size: 0.85rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.status-pending {
    background: rgba(255, 193, 7, 0.2);
    color: #ffc107;
    border: 1px solid rgba(255, 193, 7, 0.3);
}

.status-approved {
    background: rgba(40, 167, 69, 0.2);
    color: #28a745;
    border: 1px solid rgba(40, 167, 69, 0.3);
}

.status-completed {
    background: rgba(0, 255, 136, 0.2);
    color: #00ff88;
    border: 1px solid rgba(0, 255, 136, 0.3);
}

.status-rejected {
    background: rgba(220, 53, 69, 0.2);
    color: #dc3545;
    border: 1px solid rgba(220, 53, 69, 0.3);
}

.card-body {
    margin-bottom: 1rem;
}

.amount-section {
    margin-bottom: 1rem;
}

.amount-display {
    display: flex;
    align-items: baseline;
    gap: 0.5rem;
    margin-bottom: 0.5rem;
}

.currency {
    color: var(--text-secondary);
    font-size: 1.2rem;
    font-weight: 600;
}

.amount {
    color: var(--text-primary);
    font-size: 2rem;
    font-weight: 700;
}

.fee-info {
    color: var(--text-muted);
    font-size: 0.85rem;
}

.final-amount {
    color: var(--text-secondary);
    font-size: 0.85rem;
    margin-top: 0.25rem;
}

.details-section {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.detail-row {
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.detail-row .label {
    color: var(--text-secondary);
    font-size: 0.9rem;
    font-weight: 500;
}

.detail-row .value {
    color: var(--text-primary);
    font-size: 0.9rem;
    font-weight: 600;
}

.card-footer {
    border-top: 1px solid rgba(255, 255, 255, 0.1);
    padding-top: 1rem;
}

.action-buttons {
    display: flex;
    gap: 0.75rem;
    flex-wrap: wrap;
}

.btn {
    padding: 0.5rem 1rem;
    border-radius: 8px;
    font-size: 0.85rem;
    font-weight: 600;
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
}

.btn-outline-info {
    background: transparent;
    color: #17a2b8;
    border: 1px solid #17a2b8;
}

.btn-outline-info:hover {
    background: #17a2b8;
    color: white;
}

.btn-outline-warning {
    background: transparent;
    color: #ffc107;
    border: 1px solid #ffc107;
}

.btn-outline-warning:hover {
    background: #ffc107;
    color: #001f11;
}

.empty-state {
    text-align: center;
    padding: 4rem 2rem;
    color: var(--text-secondary);
}

.empty-icon {
    font-size: 4rem;
    color: var(--text-secondary);
    margin-bottom: 1.5rem;
    opacity: 0.5;
}

.empty-state h3 {
    color: var(--text-primary);
    font-size: 1.5rem;
    margin-bottom: 1rem;
}

.empty-state p {
    font-size: 1rem;
    margin-bottom: 2rem;
}

.pagination-wrapper {
    padding: 2rem;
    display: flex;
    justify-content: center;
}

.pagination {
    display: flex;
    gap: 0.5rem;
}

.pagination .page-link {
    background: rgba(255, 255, 255, 0.1);
    color: var(--text-primary);
    border: 1px solid rgba(255, 255, 255, 0.2);
    padding: 0.75rem 1rem;
    border-radius: 8px;
    text-decoration: none;
    transition: all 0.3s ease;
}

.pagination .page-link:hover {
    background: rgba(0, 255, 136, 0.2);
    border-color: rgba(0, 255, 136, 0.3);
    color: var(--text-primary);
}

.pagination .page-item.active .page-link {
    background: linear-gradient(135deg, #00ff88, #00d4aa);
    color: #001f11;
    border-color: transparent;
}

/* Mobile Responsive */
@media (max-width: 768px) {
    .withdraw-history-container {
        padding: 0 15px;
    }
    
    .page-header {
        flex-direction: column;
        text-align: center;
        padding: 1.5rem;
        gap: 1.5rem;
    }
    
    .header-content {
        width: 100%;
    }
    
    .header-actions {
        width: 100%;
        display: flex;
        justify-content: center;
    }
    
    .page-title {
        font-size: 2rem;
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .page-subtitle {
        font-size: 1rem;
    }
    
    .stats-cards {
        grid-template-columns: repeat(2, 1fr);
        gap: 1rem;
        margin-bottom: 1.5rem;
    }
    
    .stat-card {
        padding: 1rem;
        flex-direction: column;
        text-align: center;
        gap: 0.75rem;
    }
    
    .stat-icon {
        width: 50px;
        height: 50px;
        font-size: 1.2rem;
    }
    
    .stat-content h3 {
        font-size: 1.5rem;
    }
    
    .stat-content p {
        font-size: 0.9rem;
    }
    
    .withdrawals-section {
        padding: 1.5rem;
    }
    
    .withdrawals-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .withdrawal-card {
        padding: 1rem;
    }
    
    .card-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .status-badge {
        align-self: flex-start;
    }
    
    .amount-display {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.25rem;
    }
    
    .amount {
        font-size: 1.5rem;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .btn {
        justify-content: center;
        width: 100%;
    }
    
    .empty-state {
        padding: 3rem 1rem;
    }
    
    .empty-icon {
        font-size: 3rem;
    }
    
    .empty-state h3 {
        font-size: 1.3rem;
    }
    
    .empty-state p {
        font-size: 0.9rem;
    }
}

@media (max-width: 480px) {
    .withdraw-history-container {
        padding: 0 10px;
    }
    
    .page-header {
        padding: 1rem;
    }
    
    .page-title {
        font-size: 1.5rem;
    }
    
    .page-subtitle {
        font-size: 0.9rem;
    }
    
    .stats-cards {
        grid-template-columns: repeat(2, 1fr);
        gap: 0.75rem;
    }
    
    .stat-card {
        padding: 0.75rem;
        gap: 0.5rem;
    }
    
    .stat-icon {
        width: 40px;
        height: 40px;
        font-size: 1rem;
    }
    
    .stat-content h3 {
        font-size: 1.2rem;
    }
    
    .stat-content p {
        font-size: 0.8rem;
    }
    
    .withdrawals-section {
        padding: 1rem;
    }
    
    .withdrawal-card {
        padding: 0.75rem;
    }
    
    .amount {
        font-size: 1.3rem;
    }
    
    .detail-row {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.25rem;
    }
    
    .transaction-id {
        font-size: 0.9rem;
    }
    
    .transaction-method {
        font-size: 0.8rem;
    }
    
    .status-badge {
        font-size: 0.75rem;
        padding: 0.4rem 0.8rem;
    }
    
    .empty-state {
        padding: 2rem 0.75rem;
    }
    
    .empty-icon {
        font-size: 2.5rem;
    }
    
    .empty-state h3 {
        font-size: 1.1rem;
    }
    
    .empty-state p {
        font-size: 0.85rem;
    }
}

/* Extra small screens */
@media (max-width: 360px) {
    .withdraw-history-container {
        padding: 0 5px;
    }
    
    .page-header {
        padding: 0.75rem;
    }
    
    .page-title {
        font-size: 1.3rem;
    }
    
    .stats-cards {
        grid-template-columns: 1fr;
    }
    
    .stat-card {
        padding: 0.5rem;
    }
    
    .withdrawals-section {
        padding: 0.75rem;
    }
    
    .withdrawal-card {
        padding: 0.5rem;
    }
}
</style>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        // Show success message
        const btn = event.target.closest('.copy-btn');
        const originalHTML = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-check"></i> Copied!';
        btn.style.color = '#00ff88';
        
        setTimeout(() => {
            btn.innerHTML = originalHTML;
            btn.style.color = '';
        }, 2000);
    });
}
</script>
@endsection