<!-- Shah Developer Theme - InvestFlow Scripts -->
<script>
// Global variables
let sidebarOpen = false;
let notificationsOpen = false;
let userMenuOpen = false;

// Sidebar functionality
function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');
    const mainContent = document.querySelector('.main-content');
    
    sidebarOpen = !sidebarOpen;
    
    if (sidebar) {
        sidebar.classList.toggle('active', sidebarOpen);
    }
    
    if (overlay) {
        overlay.classList.toggle('active', sidebarOpen);
    }
    
    if (mainContent) {
        mainContent.classList.toggle('sidebar-open', sidebarOpen);
    }
    
    // Add haptic feedback
    if ('vibrate' in navigator) {
        navigator.vibrate(30);
    }
}

// Notification functionality
function toggleNotifications() {
    const notificationPanel = document.getElementById('notification-panel');
    
    notificationsOpen = !notificationsOpen;
    
    if (notificationPanel) {
        notificationPanel.classList.toggle('active', notificationsOpen);
    }
    
    // Close user menu if open
    if (userMenuOpen) {
        toggleUserMenu();
    }
    
    // Add haptic feedback
    if ('vibrate' in navigator) {
        navigator.vibrate(30);
    }
}

// User menu functionality
function toggleUserMenu() {
    const userMenu = document.getElementById('user-menu');
    
    userMenuOpen = !userMenuOpen;
    
    if (userMenu) {
        userMenu.classList.toggle('active', userMenuOpen);
    }
    
    // Close notifications if open
    if (notificationsOpen) {
        toggleNotifications();
    }
    
    // Add haptic feedback
    if ('vibrate' in navigator) {
        navigator.vibrate(30);
    }
}

// Utility functions
function showToast(message, type = 'info') {
    // Create toast notification
    const toast = document.createElement('div');
    toast.className = `toast toast-${type}`;
    toast.textContent = message;
    
    // Add styles
    toast.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: rgba(26, 26, 46, 0.95);
        backdrop-filter: blur(20px);
        color: white;
        padding: 1rem 1.5rem;
        border-radius: 12px;
        border: 1px solid rgba(255, 255, 255, 0.1);
        z-index: 9999;
        transform: translateX(100%);
        transition: transform 0.3s ease;
    `;
    
    // Add to page
    document.body.appendChild(toast);
    
    // Animate in
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
    }, 100);
    
    // Remove after delay
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => {
            if (document.body.contains(toast)) {
                document.body.removeChild(toast);
            }
        }, 300);
    }, 3000);
}

function formatCurrency(amount, currency = '$') {
    return `${currency}${amount.toFixed(2)}`;
}

function formatDate(date) {
    return new Date(date).toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Initialize app when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Add global event listeners
    document.addEventListener('click', (e) => {
        // Close dropdowns when clicking outside
        if (!e.target.closest('.notification-btn') && !e.target.closest('#notification-panel')) {
            if (notificationsOpen) {
                toggleNotifications();
            }
        }
        
        if (!e.target.closest('.user-avatar') && !e.target.closest('#user-menu')) {
            if (userMenuOpen) {
                toggleUserMenu();
            }
        }
    });
    
    // Add keyboard navigation
    document.addEventListener('keydown', (e) => {
        if (e.key === 'Escape') {
            if (sidebarOpen) {
                toggleSidebar();
            }
            if (notificationsOpen) {
                toggleNotifications();
            }
            if (userMenuOpen) {
                toggleUserMenu();
            }
        }
    });
    
    // Add service worker for offline functionality
    // if ('serviceWorker' in navigator) {
    //     navigator.serviceWorker.register('/sw.js')
    //         .then(registration => {
    //             console.log('Service Worker registered:', registration);
    //         })
    //         .catch(error => {
    //             console.log('Service Worker registration failed:', error);
    //         });
    // }
    
    // Initialize page-specific functionality
    const currentPage = window.location.pathname;
    if (currentPage.includes('/user/dashboard')) {
        initDashboard();
    }

    // Start notifications polling
    startTransactionNotifications();
});

// Dashboard functionality
function initDashboard() {
    // Initialize dashboard components
    setTimeout(() => {
        initCarousel();
        animateBalanceCard();
        initQuickActions();
    }, 200);
}

// Carousel functionality
function initCarousel() {
    const container = document.querySelector('.carousel-container');
    const cards = document.querySelectorAll('.carousel-card');
    const dots = document.querySelectorAll('.dot');
    
    if (!container || !cards.length) return;
    
    let currentSlide = 0;
    let carouselInterval;
    
    // Add click events to carousel cards
    cards.forEach((card, index) => {
        card.addEventListener('click', () => {
            currentSlide = index;
            updateCarousel();
        });
    });
    
    // Dot navigation
    dots.forEach((dot, index) => {
        dot.addEventListener('click', (e) => {
            e.stopPropagation();
            currentSlide = index;
            updateCarousel();
        });
    });
    
    // Touch/swipe support
    let startX = 0;
    let endX = 0;
    
    container.addEventListener('touchstart', (e) => {
        startX = e.touches[0].clientX;
    });
    
    container.addEventListener('touchend', (e) => {
        endX = e.changedTouches[0].clientX;
        handleSwipe();
    });
    
    function handleSwipe() {
        const threshold = 50;
        const diff = startX - endX;
        
        if (Math.abs(diff) > threshold) {
            if (diff > 0) {
                // Swipe left - next slide
                currentSlide = (currentSlide + 1) % cards.length;
            } else {
                // Swipe right - previous slide
                currentSlide = currentSlide === 0 ? cards.length - 1 : currentSlide - 1;
            }
            updateCarousel();
        }
    }
    
    function updateCarousel() {
        cards.forEach((card, index) => {
            card.classList.toggle('active', index === currentSlide);
        });
        
        dots.forEach((dot, index) => {
            dot.classList.toggle('active', index === currentSlide);
        });
    }
    
    // Auto-rotate carousel
    carouselInterval = setInterval(() => {
        currentSlide = (currentSlide + 1) % cards.length;
        updateCarousel();
    }, 3000);
    
    // Initial display
    updateCarousel();
}

// Animate balance card
function animateBalanceCard() {
    const balanceCard = document.querySelector('.balance-card');
    if (balanceCard) {
        balanceCard.style.opacity = '0';
        balanceCard.style.transform = 'translateY(20px)';
        
        setTimeout(() => {
            balanceCard.style.transition = 'all 0.6s ease-out';
            balanceCard.style.opacity = '1';
            balanceCard.style.transform = 'translateY(0)';
        }, 100);
    }
}

// Initialize quick actions
function initQuickActions() {
    const actionBtns = document.querySelectorAll('.action-btn');
    
    actionBtns.forEach(btn => {
        btn.addEventListener('click', (e) => {
            // Add haptic feedback
            if ('vibrate' in navigator) {
                navigator.vibrate(50);
            }
            
            // Add click animation
            btn.style.transform = 'scale(0.95)';
            setTimeout(() => {
                btn.style.transform = '';
            }, 150);
        });
    });
}

// Handle page visibility changes
document.addEventListener('visibilitychange', function() {
    if (document.hidden) {
        // Pause any running animations when page is hidden
        console.log('Page hidden');
    } else {
        // Resume animations when page is visible
        console.log('Page visible');
    }
});

// Page Transition System
class PageTransition {
    constructor() {
        this.isTransitioning = false;
        this.history = [];
        this.loadingTimeout = null;
        this.init();
    }

    init() {
        // Listen for navigation
        this.setupNavigationListeners();
        
        // Handle browser back/forward
        window.addEventListener('popstate', (e) => {
            this.handleBackNavigation();
        });

        // Handle page load events
        this.setupPageLoadListeners();
    }

    setupPageLoadListeners() {
        // Show loading animation when page starts loading
        window.addEventListener('beforeunload', () => {
            this.showLoadingAnimation();
        });

        // Hide loading animation when page is fully loaded
        window.addEventListener('load', () => {
            this.hideLoadingAnimation();
        });

        // Handle DOM content loaded
        document.addEventListener('DOMContentLoaded', () => {
            this.hideLoadingAnimation();
        });
    }

    setupNavigationListeners() {
        // Intercept all navigation links
        document.addEventListener('click', (e) => {
            const link = e.target.closest('a[href]');
            if (link && this.isInternalLink(link)) {
                e.preventDefault();
                this.navigateTo(link.href, 'forward');
            }
        });

        // Intercept form submissions
        document.addEventListener('submit', (e) => {
            const form = e.target;
            if (form.method === 'get' && form.action) {
                e.preventDefault();
                this.navigateTo(form.action + '?' + new URLSearchParams(new FormData(form)).toString(), 'forward');
            }
        });
    }

    isInternalLink(link) {
        try {
            const url = new URL(link.href);
            return url.origin === window.location.origin;
        } catch {
            return false;
        }
    }

    async navigateTo(url, direction = 'forward') {
        if (this.isTransitioning) return;
        
        this.isTransitioning = true;
        
        // Show loading animation immediately
        this.showLoadingAnimation();
        
        // Start loading timeout (fallback)
        this.loadingTimeout = setTimeout(() => {
            this.hideLoadingAnimation();
        }, 5000); // 5 second timeout
        
        // Navigate to new page immediately
        window.location.href = url;
    }

    handleBackNavigation() {
        if (this.isTransitioning) return;
        
        this.isTransitioning = true;
        
        // Show loading animation
        this.showLoadingAnimation();
        
        // Wait for animation to complete
        setTimeout(() => {
            this.isTransitioning = false;
        }, 400);
    }

    // Page transition animations removed - only loading animation needed

    wait(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    // Page transition styles removed - only loading animation needed

    setLoadingEnabled(enabled) {
        this.loadingEnabled = enabled;
    }

    setLoadingType(type) {
        this.loadingType = type;
    }

    showLoadingAnimation() {
        if (!this.loadingEnabled) return;
        
        // Create loading overlay if it doesn't exist
        let loadingOverlay = document.getElementById('page-loading-overlay');
        if (!loadingOverlay) {
            loadingOverlay = document.createElement('div');
            loadingOverlay.id = 'page-loading-overlay';
            
            // Generate loading content based on type
            let loadingContent = this.generateLoadingContent();
            
            loadingOverlay.innerHTML = `
                <div class="loading-content">
                    ${loadingContent}
                    <div class="loading-text">Loading...</div>
                </div>
            `;
            document.body.appendChild(loadingOverlay);
        }
        
        // Show loading overlay
        loadingOverlay.style.display = 'flex';
        loadingOverlay.classList.add('loading-show');
    }

    generateLoadingContent() {
        const loadingType = this.loadingType || 'spinner';
        
        switch(loadingType) {
            case 'spinner':
                return '<div class="loading-spinner"></div>';
            case 'progress':
                return '<div class="loading-progress"><div class="loading-progress-bar"></div></div>';
            case 'dots':
                return '<div class="loading-dots"><span></span><span></span><span></span></div>';
            case 'pulse':
                return '<div class="loading-pulse"></div>';
            default:
                return '<div class="loading-spinner"></div>';
        }
    }

    hideLoadingAnimation() {
        const loadingOverlay = document.getElementById('page-loading-overlay');
        if (loadingOverlay) {
            loadingOverlay.classList.remove('loading-show');
            loadingOverlay.classList.add('loading-hide');
            
            // Remove after animation completes
            setTimeout(() => {
                if (loadingOverlay.parentNode) {
                    loadingOverlay.parentNode.removeChild(loadingOverlay);
                }
            }, 300);
        }
        
        // Clear loading timeout
        if (this.loadingTimeout) {
            clearTimeout(this.loadingTimeout);
            this.loadingTimeout = null;
        }
    }
}

// Initialize loading animation system
document.addEventListener('DOMContentLoaded', () => {
    // Check if loading animation is enabled
    const loadingEnabled = <?php echo e(setting('loading_animation', 'permission') ? 'true' : 'false'); ?>;
    const loadingType = '<?php echo e(setting('loading_type', 'permission')); ?>' || 'spinner';
    
    if (loadingEnabled) {
        const transition = new PageTransition();
        transition.setLoadingEnabled(loadingEnabled);
        transition.setLoadingType(loadingType);
    }
});

// Enhanced navigation with animations
function navigateWithAnimation(url, direction = 'forward') {
    const transition = new PageTransition();
    transition.navigateTo(url, direction);
}

// Back button with animation
function goBackWithAnimation() {
    const transition = new PageTransition();
    transition.handleBackNavigation();
    window.history.back();
}

// Handle window resize
window.addEventListener('resize', function() {
    // Recalculate layouts if needed
    console.log('Window resized');
});

// Real-time-ish notifications via polling
function startTransactionNotifications() {
    const badge = document.getElementById('notification-count');
    const list = document.querySelector('#notification-panel .notification-list');
    if (!badge || !list) return;

    async function fetchLatest() {
        try {
            const res = await fetch('/user/api/transactions/latest?limit=10', { credentials: 'same-origin' });
            if (!res.ok) return;
            const data = await res.json();
            const items = Array.isArray(data.items) ? data.items : [];

            // Update badge
            badge.textContent = items.length;

            // Render list
            list.innerHTML = '';
            if (!items.length) {
                const empty = document.createElement('div');
                empty.className = 'notification-item';
                empty.innerHTML = '<div class="notification-content"><h4>No recent transactions</h4><p>Check back later.</p></div>';
                list.appendChild(empty);
                return;
            }

            items.forEach(t => {
                const item = document.createElement('div');
                item.className = 'notification-item';
                const icon = t.status === 'success' ? 'check-circle' : (t.status === 'failed' ? 'times-circle' : 'clock');
                const color = t.status === 'success' ? '#22c55e' : (t.status === 'failed' ? '#ef4444' : '#f59e0b');
                const amount = (t.final_amount ?? t.amount ?? 0).toFixed ? (t.final_amount ?? t.amount).toFixed(2) : Number(t.final_amount ?? t.amount ?? 0).toFixed(2);
                const title = `${t.type?.toUpperCase() || 'TXN'} • ${t.status?.toUpperCase() || ''}`;
                const when = new Date(t.created_at_iso || t.created_at).toLocaleString();
                item.innerHTML = `
                    <div class="notification-icon" style="color:${color}">
                        <i class="fas fa-${icon}"></i>
                    </div>
                    <div class="notification-content">
                        <h4>${title}</h4>
                        <p>Amount: ${t.currency || '$'}${amount} — Ref: ${t.tnx || '-'}</p>
                        <span class="notification-time">${when}</span>
                    </div>
                `;
                list.appendChild(item);
            });
        } catch(e) {
            // ignore
        }
    }

    fetchLatest();
    setInterval(fetchLatest, 15000);
}
</script>

<?php /**PATH /home/digitalx/blockwavetrade.com/resources/views/frontend/shahdeveloper/include/__shahdeveloper_scripts.blade.php ENDPATH**/ ?>